% ADVISOR Data file:  FC_CI330.M
%
% Data source:  Data was interpreted from Figure 5.1, p.121, SAE Publication SP-1011, 1994
% "Diesel Engine Design for the 1990's" by David F. Merrion, Detroit Diesel Corporation.
%
% Data confidence level:  no comparison has been performed
%
% Notes:   
%
% Created on:  05/20/98
% By:  Tony Markel, National Renewable Energy Laboratory, Tony_Markel@nrel.gov
%
% Revision history at end of file.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FILE ID INFO
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fc_description='Detroit Diesel Corp. Series 60 12.7L (330kW) Diesel Engine'; 
fc_version=2003; % version of ADVISOR for which the file was generated
fc_proprietary=0; % 0=> non-proprietary, 1=> proprietary, do not distribute
fc_validation=0; % 0=> no validation, 1=> data agrees with source data, 
% 2=> data matches source data and data collection methods have been verified
fc_fuel_type='Diesel';
fc_disp=12.7;  % (L), engine displacement
fc_emis=0;      % boolean 0=no emis data; 1=emis data
fc_cold=0;      % boolean 0=no cold data; 1=cold data exists
disp(['Data loaded: FC_CI330.m - ',fc_description]);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SPEED & TORQUE RANGES over which data is defined
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (rad/s), speed range of the engine
fc_map_spd=[1200 1400 1600 1800 2000 2100]*2*pi/60; 

% (N*m), torque range of the engine
fc_map_trq=[200 400 600 800 1000 1200 1400 1600 1800]; 


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FUEL USE AND EMISSIONS MAPS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (g/s), fuel use map indexed vertically by fc_map_spd and 
% horizontally by fc_map_trq
fc_fuel_map_gpkWh = [
242	220	207	201.5	198.5	197	195.5	194.5	193.5
261	220	207	200	195	192.5	190.5	189.5	189.5
273	224	207	198	194.5	192	190.5	189.5	189.5
288	228	209	200.5	196	193.5	191.5	191.5	191.5
288	238	217	207.5	203	199.5	197.5	197.5	197.5
290	244	222	210.5	204.5	202.5	199.5	199.5	199.5]; % (g/kW*hr)
% fc_fuel_map elements (1,1), (5,9), (6,8), (6,9), (7,8), (7,9) 
% have been estimated in order to complete the table.

% convert map from gpkwh to gps
[T,w]=meshgrid(fc_map_trq,fc_map_spd);
fc_map_kW=T.*w/1000;
fc_fuel_map=fc_fuel_map_gpkWh.*fc_map_kW/3600;

%fc_fuel_map=[fc_fuel_map(:,1) fc_fuel_map]; % pad with zero torque column
%fc_map_trq=[0 fc_map_trq]; % add zero torque column

% (g/s), engine out HC emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_hc_map=zeros(size(fc_fuel_map));

% (g/s), engine out HC emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_co_map=zeros(size(fc_fuel_map));

% (g/s), engine out HC emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_nox_map=zeros(size(fc_fuel_map)); 

% (g/s), engine out PM emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_pm_map=zeros(size(fc_fuel_map));

% (g/s), engine out O2 indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_o2_map=zeros(size(fc_fuel_map));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Cold Engine Maps
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fc_cold_tmp=20; %deg C
fc_fuel_map_cold=zeros(size(fc_fuel_map));
fc_hc_map_cold=zeros(size(fc_fuel_map));
fc_co_map_cold=zeros(size(fc_fuel_map));
fc_nox_map_cold=zeros(size(fc_fuel_map));
fc_pm_map_cold=zeros(size(fc_fuel_map));
%Process Cold Maps to generate Correction Factor Maps
names={'fc_fuel_map','fc_hc_map','fc_co_map','fc_nox_map','fc_pm_map'};
for i=1:length(names)
    %cold to hot raio, e.g. fc_fuel_map_c2h = fc_fuel_map_cold ./ fc_fuel_map
    eval([names{i},'_c2h=',names{i},'_cold./(',names{i},'+eps);'])
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LIMITS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (N*m), max torque curve of the engine indexed by fc_map_spd
fc_max_trq=[1959 1883 1825 1761 1585 1510]; 

% (N*m), closed throttle torque of the engine (max torque that can be absorbed)
% indexed by fc_map_spd -- correlation from JDMA
fc_ct_trq=4.448/3.281*(-fc_disp)*61.02/24 * ...
   (9*(fc_map_spd/max(fc_map_spd)).^2 + 14 * (fc_map_spd/max(fc_map_spd)));


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DEFAULT SCALING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (--), used to scale fc_map_spd to simulate a faster or slower running engine 
fc_spd_scale=1.0;
% (--), used to scale fc_map_trq to simulate a higher or lower torque engine
fc_trq_scale=1.0;
fc_pwr_scale=fc_spd_scale*fc_trq_scale;   % --  scale fc power

% user definable mass scaling function
fc_mass_scale_fun=inline('(x(1)*fc_trq_scale+x(2))*(x(3)*fc_spd_scale+x(4))*(fc_base_mass+fc_acc_mass)+fc_fuel_mass','x','fc_spd_scale','fc_trq_scale','fc_base_mass','fc_acc_mass','fc_fuel_mass');
fc_mass_scale_coef=[1 0 1 0]; % coefficients of mass scaling function


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% STUFF THAT SCALES WITH TRQ & SPD SCALES (MASS AND INERTIA)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%fc_inertia=0.1*fc_pwr_scale;   % (kg*m^2),  rotational inertia of the engine (unknown)
fc_max_pwr=(max(fc_map_spd.*fc_max_trq)/1000)*fc_pwr_scale; % kW     peak engine power

fc_base_mass=2.8*fc_max_pwr;            % (kg), mass of the engine block and head (base engine)
                                        %       assuming a mass penalty of 1.8 kg/kW from S. Sluder (ORNL) estimate of 300 lb 
fc_acc_mass=0.8*fc_max_pwr;             % kg    engine accy's, electrics, cntrl's - assumes mass penalty of 0.8 kg/kW (from 1994 OTA report, Table 3)
fc_fuel_mass=0.6*fc_max_pwr;            % kg    mass of fuel and fuel tank (from 1994 OTA report, Table 3)
fc_mass=fc_base_mass+fc_acc_mass+fc_fuel_mass; % kg  total engine/fuel system mass
%fc_mass=2500/2.2046;                   % (kg), mass of the engine
fc_ext_sarea=0.3*(fc_max_pwr/100)^0.67;       % m^2    exterior surface area of engine

fc_inertia=fc_mass*(1/3+1/3*2/3)*(0.075^2); 
% assumes 1/3 purely rotating mass, 1/3 purely oscillating, and 1/3 stationary
% and crank radius of 0.075m, 2/3 of oscilatting mass included in rotational inertia calc
% correlation from Bosch handbook p.379

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OTHER DATA
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fc_fuel_den=0.85*1000; % (g/l), density of the fuel 
fc_fuel_lhv=45300.0; % (J/g), lower heating value of the fuel

%the following was added for the new thermal modeling of the engine 12/17/98 ss and sb
fc_tstat=99;                  % C      engine coolant thermostat set temperature (typically 95 +/- 5 C)
fc_cp=500;                    % J/kgK  ave cp of engine (iron=500, Al or Mg = 1000)
fc_h_cp=500;                  % J/kgK  ave cp of hood & engine compartment (iron=500, Al or Mg = 1000)
fc_hood_sarea=1.5;            % m^2    surface area of hood/eng compt.
fc_emisv=0.8;                 %        eff emissivity of engine ext surface to hood int surface
fc_hood_emisv=0.9;            %        emissivity hood ext
fc_h_air_flow=0.0;            % kg/s   heater air flow rate (140 cfm=0.07)
fc_cl2h_eff=0.7;              % --     ave cabin heater HX eff (based on air side)
fc_c2i_th_cond=500;           % W/K    conductance btwn engine cyl & int
fc_i2x_th_cond=500;           % W/K    conductance btwn engine int & ext
fc_h2x_th_cond=10;            % W/K    conductance btwn engine & engine compartment

% calc "predicted" exh gas flow rate and engine-out (EO) temp
fc_ex_pwr_frac=[0.45 0.35];                        % --   frac of waste heat that goes to exhaust as func of engine speed
fc_exflow_map=fc_fuel_map*(1+25);                  % g/s  ex gas flow map:  for CI engines, exflow=(fuel use)*[1 + (ave A/F ratio)]
[T,w]=meshgrid(fc_map_trq, fc_map_spd);           % Nm, rad/s   full map of mech out pwr (incl 0 trq)
fc_waste_pwr_map=fc_fuel_map*fc_fuel_lhv - T.*w;   % W    tot FC waste heat = (fuel pwr) - (mech out pwr)
spd=fc_map_spd;
fc_ex_pwr_map=zeros(size(fc_waste_pwr_map));       % W   initialize size of ex pwr map
for i=1:length(spd)
 fc_ex_pwr_map(i,:)=fc_waste_pwr_map(i,:)*interp1([min(spd) max(spd)],fc_ex_pwr_frac,spd(i)); % W  trq-spd map of waste heat to exh 
end
fc_extmp_map=fc_ex_pwr_map./(fc_exflow_map*1089/1000) + 20;  % W   EO ex gas temp = Q/(MF*cp) + Tamb (assumes engine tested ~20 C)


%the following variable is not used directly in modelling and should always be equal to one
%it's used for initialization purposes
fc_eff_scale=1;

% clean up workspace
clear fc_map_kW
clear T w fc_waste_pwr_map fc_ex_pwr_map spd

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% REVISION HISTORY
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 05/20/98 (tm): file created from a_ddc127l.m 
% 7/3/98 (MC): changed fc_init_coolant_temp to fc_coolant_init_temp for
%              consistency w/ block diagrams
% 07/08/98 (tm): added zero torque padding to the fuel use and emissions maps
% 07/16/98 (SS): added variable fc_fuel_type under file id section
% 07/17/98 (tm): file renamed FC_CI330.M for consistency
% 08/28/98 (MC): added variable fc_disp under file id section
%                fc_ct_trq computed according to correlation from JDMA, 5/98
% 10/13/98 (MC): updated equation for fc_ct_trq (convert from ft-lb to Nm)
% 12/17/98 ss,sb: added 12 new variables for engine thermal modelling.
% 01/14/99 (SB): removed unneeded variables (fc_air_fuel_ratio, fc_ex_pwr_frac)
% 2/4/99: ss,sb changed fc_ext_sarea=0.3*(fc_max_pwr/100)^0.67  it was 0.3*(fc_max_pwr/100)
%		it now takes into account that surface area increases based on mass to the 2/3 power 
% 3/15/99:ss updated *_version to 2.1 from 2.0
% 7/9/99:tm added inertia calculation and cosmetic changes
% 11/03/99:ss updated version from 2.2 to 2.21
% 8/18/00:tm revised fc_inertia calculation, reduced equiv. radius from 0.15m to 0.075m
% 01/31/01: vhj added fc_cold=0, added cold map variables, added +eps to avoid dividing by zero
% 02/26/01: vhj added variable definition of fc_o2_map (used in NOx absorber emis.)
% 7/30/01:tm added user definable mass scaling function mass=f(fc_spd_scale,fc_trq_scale,fc_base_mass,fc_acc_mass,fc_fuel_mass)
% Begin added by ADVISOR 2002 converter: 11-Apr-2003
a=37;

b=34;

ex_spd=[89.01 94.25 99.48 104.7 110 115.2 120.4 125.7 130.9 136.1 141.4 146.6 151.8 157.1 162.3 167.6 172.8 178 183.3 188.5 193.7 199 204.2 209.4 214.7 219.9 225.1 230.4 235.6 240.9 246.1 251.3 256.6 261.8 267 272.3 277.5 282.7 288 293.2 298.5 303.7 308.9 314.2 319.4 324.6 329.9 335.1 340.3 345.6 350.8 356 361.3 366.5 371.8 377 382.2 387.5 392.7 397.9 403.2 408.4 413.6 418.9];

fc_co_map_gpkWh=[40.44 36.77 18.73 13 10.35 8.888 7.818 6.792 5.729 4.733 3.899 3.25 2.773 2.414 2.155 1.999 2.013 2.353 3.268 4.861 6.964 9.059 10.64 11.42 11.51 11.17 10.64 10.07 9.473 8.825 8.011 6.992 5.931 5.189;39.11 35.56 18.17 12.51 9.819 8.261 7.137 6.118 5.128 4.214 3.465 2.889 2.465 2.146 1.916 1.777 1.789 2.092 2.905 4.321 6.19 8.052 9.451 10.14 10.19 9.829 9.261 8.603 7.931 7.251 6.504 5.651 4.798 4.207;40.39 36.72 18.76 12.8 9.813 8.023 6.728 5.644 4.666 3.815 3.129 2.605 2.223 1.935 1.728 1.603 1.613 1.882 2.612 3.878 5.553 7.218 8.465 9.053 9.052 8.625 7.95 7.143 6.325 5.57 4.872 4.191 3.569 3.147;42.55 38.68 19.76 13.34 10.02 7.948 6.478 5.308 4.322 3.504 2.868 2.389 2.036 1.773 1.581 1.466 1.47 1.709 2.351 3.477 4.965 6.439 7.525 8.006 7.926 7.421 6.644 5.72 4.801 4.006 3.373 2.863 2.452 2.188;44.77 40.7 20.71 13.85 10.23 7.928 6.304 5.059 4.068 3.278 2.675 2.228 1.898 1.655 1.476 1.364 1.355 1.549 2.094 3.056 4.325 5.574 6.476 6.828 6.665 6.109 5.305 4.375 3.476 2.734 2.206 1.849 1.607 1.46;46.82 42.56 21.51 14.28 10.43 7.943 6.196 4.893 3.896 3.125 2.551 2.124 1.814 1.581 1.41 1.293 1.265 1.404 1.825 2.579 3.579 4.559 5.238 5.453 5.231 4.687 3.961 3.163 2.421 1.835 1.447 1.218 1.089 1.015;48.69 44.27 22.28 14.69 10.65 8.005 6.14 4.782 3.776 3.027 2.478 2.068 1.772 1.546 1.379 1.258 1.207 1.276 1.552 2.069 2.762 3.436 3.883 3.978 3.749 3.296 2.736 2.163 1.657 1.274 1.032 0.9009 0.8354 0.798;50.36 45.78 22.97 15.11 10.88 8.08 6.103 4.685 3.677 2.953 2.431 2.044 1.761 1.547 1.384 1.26 1.188 1.193 1.33 1.617 2.015 2.4 2.64 2.655 2.465 2.148 1.788 1.449 1.17 0.97 0.8486 0.7842 0.7501 0.7274;51.43 46.75 23.46 15.39 11.01 8.101 6.033 4.573 3.57 2.881 2.393 2.037 1.775 1.575 1.422 1.302 1.214 1.175 1.203 1.311 1.477 1.638 1.725 1.699 1.572 1.39 1.202 1.04 0.9203 0.8405 0.7945 0.7679 0.7488 0.73;51.91 47.19 23.64 15.48 11.02 8.025 5.912 4.432 3.453 2.804 2.36 2.039 1.8 1.618 1.481 1.372 1.283 1.218 1.183 1.181 1.205 1.227 1.226 1.183 1.107 1.019 0.9385 0.8784 0.8406 0.8188 0.8074 0.7987 0.7868 0.7693;52.28 47.53 23.75 15.5 10.97 7.939 5.794 4.319 3.36 2.745 2.336 2.044 1.824 1.662 1.542 1.444 1.363 1.291 1.229 1.176 1.133 1.091 1.046 0.9983 0.9509 0.9095 0.8788 0.8609 0.8523 0.8501 0.849 0.8452 0.8364 0.8205;53.44 48.58 24.18 15.71 11.08 7.991 5.81 4.316 3.353 2.743 2.343 2.06 1.851 1.698 1.588 1.502 1.428 1.358 1.289 1.222 1.156 1.093 1.034 0.9829 0.9431 0.9171 0.9024 0.8956 0.8951 0.8955 0.8959 0.8937 0.885 0.8687;55.76 50.69 25.17 16.3 11.49 8.287 6.031 4.481 3.474 2.829 2.401 2.095 1.876 1.722 1.612 1.53 1.46 1.393 1.327 1.26 1.191 1.123 1.062 1.01 0.9731 0.9504 0.9402 0.9373 0.9373 0.9382 0.9382 0.9341 0.9239 0.9059;58.97 53.61 26.56 17.21 12.14 8.781 6.42 4.785 3.704 2.989 2.502 2.154 1.906 1.737 1.62 1.534 1.466 1.402 1.34 1.275 1.209 1.143 1.083 1.034 0.9992 0.9795 0.9715 0.9722 0.9746 0.9768 0.9756 0.9683 0.9554 0.9345;61.97 56.34 27.9 18.07 12.76 9.271 6.819 5.108 3.953 3.163 2.607 2.209 1.931 1.741 1.614 1.523 1.454 1.393 1.333 1.273 1.21 1.148 1.092 1.045 1.015 0.9986 0.9957 0.9989 1.004 1.008 1.005 0.9949 0.9775 0.9542;63.72 57.92 28.65 18.54 13.09 9.525 7.037 5.301 4.109 3.276 2.675 2.237 1.935 1.732 1.597 1.502 1.432 1.374 1.319 1.264 1.205 1.147 1.093 1.051 1.023 1.012 1.012 1.019 1.027 1.031 1.027 1.013 0.9915 0.9655;63.64 57.86 28.55 18.4 12.95 9.407 6.966 5.277 4.113 3.282 2.673 2.226 1.916 1.71 1.573 1.478 1.409 1.354 1.304 1.252 1.197 1.142 1.091 1.052 1.027 1.018 1.022 1.032 1.043 1.047 1.041 1.024 0.9997 0.9714;62.03 56.39 27.73 17.77 12.42 8.978 6.643 5.057 3.971 3.194 2.613 2.182 1.881 1.681 1.547 1.456 1.391 1.339 1.293 1.244 1.193 1.139 1.091 1.052 1.029 1.023 1.03 1.044 1.058 1.063 1.056 1.036 1.009 0.9786;59.54 54.13 26.51 16.87 11.69 8.391 6.196 4.743 3.762 3.056 2.524 2.126 1.845 1.656 1.53 1.443 1.381 1.333 1.288 1.242 1.191 1.139 1.091 1.054 1.033 1.028 1.038 1.056 1.073 1.081 1.073 1.052 1.023 0.9918;56.88 51.71 25.23 15.97 10.97 7.829 5.773 4.441 3.557 2.928 2.447 2.085 1.825 1.647 1.527 1.444 1.382 1.334 1.289 1.242 1.191 1.139 1.093 1.058 1.038 1.035 1.047 1.068 1.089 1.098 1.091 1.069 1.039 1.007;54.41 49.46 24.09 15.18 10.4 7.393 5.455 4.22 3.41 2.84 2.403 2.071 1.829 1.659 1.541 1.454 1.389 1.336 1.289 1.24 1.189 1.139 1.096 1.062 1.044 1.042 1.054 1.076 1.098 1.108 1.101 1.079 1.048 1.016;52.23 47.48 23.11 14.57 9.977 7.11 5.264 4.095 3.334 2.802 2.398 2.089 1.859 1.691 1.566 1.471 1.397 1.336 1.284 1.235 1.186 1.138 1.099 1.069 1.053 1.05 1.061 1.082 1.1 1.108 1.099 1.076 1.045 1.014;50.24 45.68 22.25 14.06 9.665 6.927 5.164 4.038 3.307 2.799 2.416 2.121 1.899 1.729 1.596 1.489 1.403 1.333 1.277 1.227 1.181 1.139 1.104 1.08 1.066 1.064 1.073 1.088 1.101 1.103 1.089 1.064 1.033 1.002;48.19 43.81 21.37 13.55 9.37 6.769 5.085 4.006 3.294 2.799 2.431 2.149 1.93 1.759 1.62 1.502 1.406 1.329 1.269 1.219 1.176 1.141 1.115 1.097 1.088 1.088 1.095 1.104 1.108 1.101 1.08 1.05 1.017 0.9855;45.6 41.46 20.27 12.9 8.986 6.552 4.971 3.944 3.261 2.782 2.426 2.154 1.941 1.77 1.628 1.506 1.405 1.324 1.262 1.214 1.176 1.149 1.132 1.125 1.125 1.13 1.135 1.138 1.131 1.112 1.081 1.044 1.005 0.9706;42.23 38.39 18.81 12.03 8.448 6.23 4.782 3.833 3.191 2.737 2.397 2.136 1.931 1.764 1.623 1.503 1.403 1.323 1.262 1.216 1.183 1.164 1.159 1.165 1.178 1.19 1.197 1.193 1.175 1.143 1.1 1.051 1.002 0.9621;38.25 34.77 17.08 10.99 7.797 5.829 4.544 3.688 3.102 2.682 2.361 2.113 1.915 1.754 1.618 1.505 1.41 1.334 1.274 1.23 1.203 1.193 1.199 1.219 1.244 1.265 1.274 1.266 1.238 1.194 1.137 1.073 1.011 0.9627;34.51 31.37 15.44 9.993 7.176 5.457 4.328 3.569 3.036 2.647 2.346 2.109 1.919 1.763 1.635 1.528 1.441 1.37 1.314 1.273 1.25 1.247 1.261 1.288 1.319 1.344 1.354 1.342 1.307 1.252 1.183 1.105 1.031 0.9761;31.89 28.99 14.28 9.302 6.758 5.223 4.217 3.528 3.034 2.664 2.376 2.146 1.962 1.812 1.69 1.593 1.513 1.448 1.395 1.358 1.34 1.34 1.353 1.376 1.402 1.423 1.429 1.411 1.369 1.307 1.228 1.142 1.061 1.001;30.91 28.1 13.85 9.051 6.634 5.192 4.247 3.591 3.111 2.747 2.46 2.232 2.048 1.902 1.789 1.7 1.628 1.571 1.523 1.49 1.475 1.473 1.479 1.489 1.499 1.506 1.5 1.472 1.422 1.351 1.266 1.177 1.095 1.035;31.17 28.33 13.97 9.151 6.743 5.312 4.377 3.722 3.24 2.868 2.576 2.344 2.161 2.017 1.909 1.827 1.765 1.715 1.676 1.649 1.635 1.629 1.625 1.616 1.606 1.592 1.569 1.528 1.466 1.387 1.297 1.208 1.13 1.074;31.93 29.02 14.32 9.389 6.931 5.48 4.527 3.862 3.368 2.988 2.689 2.454 2.269 2.128 2.022 1.944 1.887 1.844 1.813 1.792 1.78 1.771 1.756 1.733 1.705 1.672 1.631 1.576 1.504 1.416 1.324 1.237 1.165 1.112;32.52 29.57 14.59 9.572 7.071 5.594 4.628 3.952 3.453 3.071 2.771 2.536 2.353 2.211 2.103 2.023 1.965 1.924 1.897 1.88 1.871 1.861 1.842 1.813 1.773 1.727 1.674 1.609 1.53 1.441 1.349 1.267 1.2 1.15;32.75 29.77 14.69 9.631 7.112 5.623 4.655 3.983 3.488 3.11 2.816 2.586 2.405 2.262 2.148 2.06 1.994 1.946 1.916 1.9 1.892 1.885 1.87 1.84 1.799 1.747 1.687 1.619 1.541 1.456 1.372 1.297 1.236 1.189;32.69 29.72 14.63 9.583 7.067 5.586 4.627 3.964 3.481 3.115 2.832 2.607 2.43 2.286 2.166 2.066 1.985 1.924 1.885 1.866 1.861 1.859 1.85 1.826 1.787 1.734 1.674 1.607 1.536 1.461 1.388 1.322 1.267 1.223;32.41 29.46 14.49 9.467 6.973 5.508 4.564 3.916 3.447 3.094 2.821 2.605 2.431 2.287 2.162 2.051 1.956 1.882 1.833 1.808 1.804 1.808 1.807 1.79 1.754 1.704 1.645 1.582 1.518 1.453 1.392 1.335 1.286 1.244;31.92 29.02 14.26 9.305 6.846 5.406 4.482 3.85 3.393 3.05 2.786 2.577 2.408 2.264 2.137 2.021 1.919 1.837 1.78 1.753 1.75 1.757 1.761 1.748 1.715 1.667 1.61 1.55 1.491 1.434 1.379 1.328 1.283 1.243];

fc_exflow_map_cold=[14.09 14.1 14.1 14.09 14.09 14.21 14.59 15.34 16.39 17.57 18.66 19.58 20.28 20.81 21.2 21.49 21.72 21.93 22.15 22.38 22.63 22.88 23.09 23.24 23.35 23.45 23.62 23.93 24.39 24.92 25.38 25.69 25.85 25.9;13.64 13.66 13.69 13.72 13.8 14.01 14.48 15.29 16.38 17.57 18.67 19.58 20.29 20.82 21.21 21.5 21.73 21.94 22.16 22.4 22.66 22.91 23.12 23.28 23.41 23.56 23.88 24.48 25.36 26.35 27.23 27.81 28.11 28.2;12.87 12.91 12.98 13.1 13.32 13.69 14.32 15.25 16.41 17.62 18.72 19.63 20.34 20.89 21.29 21.59 21.81 22.02 22.26 22.53 22.82 23.1 23.33 23.51 23.68 23.95 24.55 25.64 27.22 28.99 30.54 31.58 32.09 32.26;12.11 12.15 12.26 12.48 12.85 13.43 14.27 15.35 16.56 17.79 18.88 19.81 20.54 21.13 21.55 21.86 22.09 22.32 22.6 22.95 23.35 23.72 24.02 24.25 24.52 24.99 25.94 27.56 29.85 32.36 34.55 36.01 36.73 36.97;11.73 11.75 11.87 12.15 12.67 13.46 14.5 15.72 16.99 18.21 19.28 20.22 21.01 21.65 22.13 22.47 22.72 23 23.39 23.93 24.55 25.11 25.54 25.89 26.32 27.04 28.32 30.33 33.02 35.91 38.41 40.05 40.87 41.14;11.96 11.97 12.06 12.37 13 13.95 15.16 16.48 17.77 18.96 20.02 20.97 21.82 22.54 23.1 23.51 23.83 24.21 24.8 25.64 26.59 27.46 28.11 28.64 29.27 30.23 31.73 33.84 36.47 39.19 41.49 42.99 43.73 43.98;12.82 12.8 12.88 13.2 13.88 14.95 16.25 17.63 18.92 20.08 21.11 22.09 22.99 23.8 24.45 24.96 25.39 25.93 26.78 27.98 29.34 30.58 31.52 32.27 33.1 34.23 35.78 37.71 39.9 42.04 43.8 44.93 45.48 45.66;14.14 14.12 14.19 14.52 15.24 16.35 17.71 19.09 20.36 21.49 22.53 23.53 24.49 25.37 26.12 26.75 27.32 28.05 29.15 30.68 32.42 34.04 35.31 36.31 37.31 38.51 39.95 41.55 43.17 44.64 45.79 46.51 46.87 46.99;15.71 15.7 15.79 16.14 16.88 18.02 19.38 20.76 21.99 23.11 24.18 25.22 26.23 27.18 28.03 28.78 29.5 30.39 31.66 33.4 35.41 37.36 38.97 40.25 41.4 42.58 43.85 45.12 46.29 47.26 47.99 48.46 48.7 48.8;17.41 17.42 17.53 17.9 18.66 19.82 21.19 22.55 23.76 24.88 25.99 27.1 28.17 29.19 30.12 30.99 31.84 32.83 34.16 35.97 38.13 40.35 42.31 43.89 45.18 46.34 47.44 48.49 49.41 50.14 50.69 51.07 51.3 51.41;19.19 19.22 19.34 19.73 20.51 21.69 23.07 24.42 25.63 26.76 27.91 29.09 30.24 31.33 32.34 33.31 34.25 35.3 36.63 38.42 40.65 43.09 45.38 47.26 48.69 49.82 50.81 51.73 52.54 53.21 53.74 54.14 54.41 54.55;21.18 21.21 21.34 21.74 22.53 23.72 25.11 26.47 27.68 28.84 30.04 31.3 32.52 33.67 34.76 35.8 36.83 37.92 39.26 41.05 43.36 46 48.56 50.67 52.19 53.29 54.19 55.02 55.78 56.42 56.95 57.38 57.7 57.86;23.51 23.53 23.66 24.06 24.85 26.04 27.43 28.8 30.04 31.25 32.52 33.83 35.11 36.3 37.45 38.57 39.69 40.85 42.26 44.13 46.59 49.43 52.19 54.42 55.95 56.97 57.74 58.45 59.12 59.7 60.19 60.6 60.91 61.08;26.21 26.24 26.37 26.75 27.52 28.69 30.07 31.45 32.75 34.05 35.39 36.75 38.05 39.26 40.46 41.66 42.89 44.19 45.74 47.8 50.48 53.53 56.43 58.67 60.12 60.99 61.59 62.13 62.64 63.1 63.49 63.82 64.07 64.21;29.03 29.06 29.18 29.56 30.3 31.43 32.79 34.19 35.56 36.95 38.37 39.77 41.07 42.3 43.54 44.85 46.22 47.69 49.44 51.72 54.65 57.89 60.87 63.06 64.37 65.06 65.48 65.83 66.15 66.43 66.67 66.88 67.03 67.12;31.66 31.69 31.81 32.17 32.89 34 35.33 36.76 38.19 39.66 41.14 42.57 43.88 45.13 46.44 47.88 49.44 51.11 53.06 55.54 58.64 62.01 65 67.1 68.29 68.84 69.11 69.28 69.41 69.49 69.56 69.61 69.65 69.68;34.04 34.07 34.19 34.54 35.25 36.35 37.68 39.12 40.6 42.13 43.66 45.12 46.44 47.72 49.12 50.72 52.49 54.37 56.5 59.09 62.25 65.6 68.5 70.48 71.56 72.03 72.22 72.27 72.25 72.16 72.05 71.96 71.89 71.86;36.47 36.5 36.63 37 37.74 38.85 40.21 41.67 43.16 44.72 46.27 47.76 49.12 50.46 51.96 53.72 55.68 57.76 60.01 62.64 65.73 68.9 71.59 73.38 74.35 74.78 74.96 74.96 74.83 74.61 74.37 74.16 74.02 73.95;39.33 39.37 39.53 39.97 40.8 42 43.41 44.87 46.32 47.84 49.39 50.92 52.35 53.79 55.39 57.25 59.31 61.49 63.81 66.41 69.33 72.2 74.55 76.09 76.92 77.31 77.49 77.48 77.32 77.04 76.75 76.48 76.3 76.21;42.77 42.84 43.08 43.68 44.72 46.12 47.61 49.01 50.31 51.67 53.17 54.76 56.35 57.95 59.64 61.49 63.5 65.62 67.89 70.39 73.08 75.6 77.55 78.78 79.44 79.77 79.93 79.92 79.77 79.5 79.21 78.95 78.77 78.68;46.97 47.08 47.45 48.29 49.65 51.31 52.89 54.16 55.19 56.29 57.69 59.36 61.15 62.93 64.66 66.4 68.16 70.03 72.09 74.41 76.81 78.96 80.52 81.45 81.92 82.17 82.28 82.27 82.15 81.93 81.69 81.47 81.32 81.25;52.05 52.21 52.72 53.81 55.48 57.39 59.03 60.11 60.83 61.64 62.92 64.65 66.61 68.53 70.23 71.73 73.13 74.61 76.34 78.36 80.42 82.19 83.4 84.07 84.39 84.55 84.61 84.59 84.49 84.33 84.13 83.94 83.82 83.77;58.03 58.22 58.82 60.03 61.84 63.83 65.43 66.37 66.91 67.57 68.77 70.51 72.5 74.38 75.92 77.1 78.09 79.16 80.52 82.19 83.88 85.27 86.17 86.62 86.83 86.92 86.96 86.94 86.87 86.73 86.57 86.41 86.31 86.26;64.17 64.38 64.99 66.18 67.91 69.76 71.26 72.17 72.76 73.5 74.73 76.39 78.22 79.86 81.1 81.93 82.57 83.33 84.4 85.75 87.1 88.14 88.75 89.04 89.16 89.24 89.28 89.28 89.22 89.1 88.94 88.78 88.67 88.61;69.51 69.76 70.37 71.44 72.91 74.48 75.82 76.82 77.69 78.73 80.1 81.66 83.19 84.44 85.28 85.77 86.16 86.75 87.69 88.86 89.95 90.7 91.07 91.22 91.3 91.41 91.5 91.54 91.5 91.37 91.17 90.95 90.76 90.65;73.27 73.6 74.28 75.27 76.49 77.77 78.97 80.11 81.35 82.78 84.33 85.82 87.05 87.89 88.34 88.53 88.78 89.36 90.31 91.43 92.35 92.88 93.08 93.14 93.25 93.43 93.62 93.72 93.67 93.49 93.19 92.81 92.44 92.21;75.58 76.05 76.88 77.89 78.96 80 81.08 82.35 83.9 85.66 87.41 88.88 89.9 90.42 90.55 90.53 90.71 91.35 92.4 93.53 94.35 94.73 94.81 94.85 95.03 95.35 95.66 95.83 95.79 95.54 95.08 94.46 93.83 93.43;77.07 77.67 78.66 79.74 80.72 81.63 82.65 84 85.75 87.73 89.62 91.09 92.01 92.34 92.27 92.12 92.29 93.02 94.15 95.28 96.02 96.29 96.31 96.38 96.67 97.13 97.56 97.8 97.77 97.45 96.84 95.98 95.11 94.55;78.42 79.07 80.1 81.18 82.11 82.98 84.03 85.48 87.38 89.46 91.39 92.86 93.71 93.95 93.79 93.61 93.81 94.58 95.72 96.78 97.43 97.63 97.64 97.78 98.17 98.71 99.21 99.48 99.47 99.11 98.42 97.45 96.47 95.83;79.9 80.48 81.4 82.35 83.24 84.17 85.41 87.09 89.13 91.23 93.09 94.46 95.23 95.44 95.29 95.15 95.39 96.14 97.17 98.09 98.63 98.82 98.89 99.1 99.53 100.1 100.5 100.8 100.7 100.4 99.72 98.8 97.89 97.3;81.46 81.91 82.64 83.44 84.31 85.42 86.99 88.98 91.18 93.23 94.9 96.06 96.7 96.89 96.81 96.77 97.04 97.69 98.52 99.25 99.71 99.94 100.1 100.4 100.8 101.2 101.6 101.7 101.7 101.3 100.7 99.97 99.25 98.8;82.96 83.35 84 84.76 85.71 87.03 88.9 91.15 93.43 95.34 96.72 97.6 98.07 98.23 98.25 98.31 98.58 99.08 99.69 100.2 100.7 101 101.2 101.5 101.8 102.2 102.4 102.4 102.3 102 101.5 100.9 100.4 100.2;84.33 84.85 85.69 86.64 87.76 89.23 91.2 93.46 95.59 97.23 98.28 98.87 99.17 99.32 99.41 99.55 99.8 100.2 100.6 101 101.4 101.8 102.1 102.4 102.7 102.9 103 102.9 102.8 102.5 102.1 101.7 101.4 101.2;85.61 86.44 87.75 89.14 90.49 91.98 93.74 95.65 97.38 98.63 99.35 99.7 99.88 100 100.2 100.3 100.6 100.8 101.1 101.4 101.9 102.3 102.7 103 103.2 103.3 103.3 103.3 103.1 102.9 102.6 102.3 102.1 102;86.76 87.99 89.91 91.84 93.45 94.82 96.16 97.51 98.68 99.49 99.92 100.1 100.2 100.3 100.5 100.7 100.9 101.1 101.3 101.7 102.1 102.6 103 103.3 103.4 103.5 103.5 103.4 103.3 103.2 103 102.8 102.7 102.6;87.67 89.26 91.72 94.13 95.94 97.16 98.07 98.85 99.5 99.93 100.2 100.3 100.3 100.5 100.6 100.8 101 101.2 101.4 101.7 102.2 102.6 103.1 103.4 103.5 103.6 103.6 103.5 103.5 103.4 103.3 103.2 103.1 103.1;88.17 89.97 92.75 95.43 97.34 98.47 99.11 99.55 99.89 100.1 100.2 100.3 100.4 100.5 100.7 100.9 101.1 101.2 101.4 101.7 102.2 102.7 103.1 103.4 103.5 103.6 103.6 103.6 103.5 103.5 103.4 103.4 103.3 103.3];

fc_exflow_vol_map=[0.011 0.011 0.012 0.012 0.013 0.014 0.016 0.019 0.022 0.026 0.028 0.031 0.033 0.034 0.036 0.037 0.039 0.04 0.041 0.043 0.044 0.045 0.046 0.047 0.047 0.047 0.048 0.049 0.05 0.051 0.054 0.058 0.063 0.066;0.011 0.011 0.012 0.012 0.013 0.014 0.016 0.019 0.022 0.026 0.028 0.031 0.033 0.034 0.036 0.037 0.039 0.04 0.041 0.043 0.044 0.046 0.046 0.047 0.047 0.048 0.049 0.05 0.052 0.054 0.057 0.061 0.065 0.068;0.011 0.012 0.012 0.012 0.013 0.015 0.017 0.02 0.023 0.026 0.029 0.031 0.033 0.035 0.036 0.038 0.039 0.04 0.042 0.043 0.044 0.046 0.046 0.047 0.048 0.049 0.051 0.053 0.057 0.06 0.063 0.067 0.07 0.073;0.012 0.012 0.012 0.013 0.014 0.015 0.017 0.02 0.023 0.026 0.029 0.031 0.033 0.035 0.037 0.038 0.039 0.041 0.042 0.044 0.045 0.046 0.047 0.048 0.049 0.051 0.053 0.058 0.062 0.067 0.07 0.074 0.076 0.078;0.013 0.013 0.014 0.014 0.015 0.017 0.019 0.021 0.024 0.027 0.03 0.032 0.034 0.036 0.038 0.039 0.041 0.042 0.044 0.045 0.047 0.048 0.049 0.05 0.052 0.054 0.058 0.063 0.068 0.073 0.077 0.08 0.081 0.082;0.015 0.015 0.015 0.016 0.017 0.018 0.02 0.023 0.026 0.029 0.031 0.034 0.036 0.038 0.04 0.041 0.043 0.045 0.046 0.048 0.049 0.051 0.052 0.054 0.056 0.059 0.063 0.068 0.073 0.078 0.082 0.084 0.085 0.085;0.016 0.017 0.017 0.018 0.019 0.02 0.022 0.025 0.028 0.031 0.034 0.036 0.039 0.041 0.043 0.044 0.046 0.048 0.05 0.052 0.054 0.056 0.058 0.059 0.062 0.065 0.069 0.073 0.078 0.082 0.085 0.087 0.087 0.088;0.018 0.018 0.019 0.019 0.021 0.022 0.025 0.028 0.031 0.034 0.037 0.039 0.042 0.044 0.046 0.048 0.05 0.052 0.055 0.057 0.06 0.062 0.064 0.066 0.069 0.072 0.075 0.079 0.083 0.086 0.088 0.089 0.09 0.09;0.02 0.02 0.02 0.021 0.022 0.025 0.027 0.03 0.034 0.037 0.04 0.043 0.045 0.048 0.05 0.052 0.055 0.057 0.06 0.063 0.065 0.068 0.071 0.073 0.076 0.079 0.082 0.085 0.088 0.09 0.092 0.093 0.094 0.094;0.021 0.021 0.021 0.022 0.024 0.027 0.03 0.033 0.036 0.04 0.043 0.046 0.049 0.051 0.054 0.057 0.059 0.062 0.065 0.068 0.071 0.074 0.078 0.08 0.083 0.086 0.089 0.091 0.094 0.096 0.098 0.099 0.1 0.101;0.022 0.023 0.023 0.024 0.026 0.029 0.033 0.036 0.039 0.042 0.046 0.049 0.052 0.055 0.058 0.061 0.064 0.067 0.07 0.073 0.076 0.08 0.084 0.087 0.09 0.093 0.095 0.098 0.101 0.103 0.105 0.107 0.108 0.109;0.025 0.025 0.025 0.026 0.028 0.032 0.035 0.039 0.043 0.046 0.049 0.052 0.056 0.059 0.062 0.065 0.068 0.072 0.075 0.078 0.082 0.086 0.09 0.094 0.097 0.1 0.102 0.105 0.108 0.111 0.113 0.115 0.117 0.117;0.027 0.028 0.028 0.029 0.031 0.035 0.039 0.043 0.046 0.049 0.053 0.056 0.06 0.063 0.066 0.07 0.073 0.077 0.08 0.083 0.088 0.092 0.097 0.101 0.104 0.107 0.11 0.113 0.116 0.119 0.122 0.124 0.125 0.126;0.031 0.031 0.031 0.033 0.035 0.038 0.043 0.047 0.05 0.054 0.057 0.061 0.064 0.068 0.071 0.075 0.079 0.082 0.086 0.09 0.095 0.1 0.105 0.109 0.112 0.115 0.119 0.122 0.126 0.129 0.131 0.133 0.135 0.135;0.034 0.034 0.035 0.036 0.039 0.042 0.047 0.051 0.055 0.058 0.062 0.066 0.069 0.073 0.076 0.08 0.084 0.088 0.092 0.097 0.102 0.108 0.113 0.117 0.121 0.124 0.127 0.131 0.135 0.138 0.141 0.142 0.144 0.144;0.038 0.038 0.038 0.04 0.042 0.046 0.051 0.055 0.059 0.063 0.066 0.07 0.074 0.077 0.08 0.084 0.088 0.093 0.098 0.103 0.109 0.115 0.121 0.125 0.128 0.132 0.136 0.14 0.143 0.147 0.149 0.151 0.152 0.152;0.041 0.041 0.041 0.043 0.046 0.05 0.055 0.059 0.063 0.067 0.07 0.074 0.077 0.081 0.084 0.088 0.093 0.097 0.103 0.109 0.115 0.122 0.127 0.132 0.136 0.139 0.143 0.148 0.151 0.154 0.156 0.158 0.158 0.158;0.044 0.044 0.045 0.047 0.051 0.055 0.059 0.063 0.067 0.07 0.074 0.078 0.081 0.084 0.088 0.092 0.097 0.102 0.108 0.115 0.122 0.128 0.134 0.138 0.142 0.146 0.151 0.155 0.158 0.161 0.163 0.164 0.164 0.164;0.048 0.048 0.049 0.052 0.055 0.06 0.064 0.068 0.072 0.075 0.079 0.082 0.086 0.089 0.092 0.097 0.101 0.107 0.113 0.121 0.128 0.134 0.14 0.144 0.148 0.153 0.158 0.162 0.166 0.168 0.169 0.17 0.17 0.17;0.053 0.053 0.055 0.057 0.061 0.065 0.07 0.074 0.077 0.081 0.085 0.089 0.092 0.095 0.099 0.103 0.108 0.114 0.12 0.127 0.135 0.141 0.146 0.151 0.155 0.16 0.165 0.169 0.173 0.175 0.176 0.177 0.177 0.176;0.059 0.059 0.061 0.063 0.067 0.072 0.077 0.081 0.085 0.089 0.094 0.098 0.101 0.104 0.108 0.112 0.117 0.122 0.128 0.135 0.142 0.148 0.153 0.158 0.162 0.168 0.173 0.177 0.18 0.182 0.183 0.183 0.183 0.183;0.066 0.067 0.068 0.071 0.075 0.08 0.085 0.09 0.095 0.1 0.104 0.108 0.112 0.115 0.119 0.123 0.128 0.133 0.138 0.144 0.15 0.155 0.161 0.166 0.171 0.176 0.18 0.184 0.187 0.189 0.19 0.19 0.19 0.19;0.075 0.075 0.077 0.08 0.084 0.09 0.095 0.101 0.106 0.112 0.117 0.121 0.124 0.128 0.132 0.136 0.14 0.145 0.149 0.154 0.159 0.165 0.17 0.175 0.181 0.185 0.19 0.193 0.195 0.197 0.197 0.197 0.197 0.197;0.085 0.086 0.087 0.091 0.096 0.101 0.108 0.114 0.119 0.125 0.129 0.134 0.137 0.141 0.146 0.15 0.154 0.159 0.162 0.166 0.171 0.175 0.181 0.187 0.193 0.198 0.201 0.204 0.205 0.206 0.206 0.205 0.204 0.204;0.097 0.098 0.1 0.103 0.108 0.115 0.121 0.127 0.133 0.138 0.143 0.147 0.151 0.155 0.16 0.165 0.17 0.174 0.177 0.181 0.184 0.189 0.196 0.203 0.209 0.214 0.217 0.218 0.218 0.218 0.217 0.215 0.213 0.211;0.112 0.113 0.114 0.118 0.123 0.129 0.135 0.141 0.146 0.151 0.156 0.16 0.165 0.17 0.176 0.181 0.186 0.191 0.194 0.197 0.201 0.206 0.214 0.222 0.229 0.235 0.237 0.238 0.237 0.235 0.231 0.227 0.222 0.219;0.128 0.129 0.131 0.134 0.139 0.144 0.15 0.155 0.16 0.164 0.169 0.174 0.179 0.185 0.19 0.196 0.202 0.207 0.212 0.215 0.219 0.225 0.234 0.244 0.252 0.258 0.261 0.261 0.259 0.256 0.25 0.242 0.234 0.229;0.143 0.144 0.146 0.149 0.153 0.158 0.163 0.167 0.171 0.175 0.18 0.185 0.19 0.196 0.203 0.209 0.215 0.221 0.226 0.231 0.236 0.243 0.253 0.264 0.273 0.28 0.283 0.284 0.282 0.278 0.27 0.26 0.249 0.241;0.153 0.154 0.156 0.159 0.163 0.167 0.172 0.175 0.179 0.182 0.186 0.191 0.197 0.203 0.21 0.216 0.223 0.229 0.235 0.241 0.247 0.255 0.266 0.277 0.287 0.294 0.298 0.3 0.299 0.296 0.288 0.277 0.265 0.258;0.154 0.155 0.158 0.162 0.166 0.17 0.174 0.177 0.18 0.184 0.188 0.193 0.198 0.205 0.211 0.218 0.224 0.231 0.237 0.244 0.251 0.26 0.27 0.281 0.291 0.298 0.303 0.306 0.307 0.305 0.301 0.292 0.282 0.276;0.149 0.151 0.154 0.158 0.162 0.167 0.171 0.174 0.178 0.182 0.186 0.191 0.196 0.202 0.209 0.215 0.221 0.228 0.235 0.242 0.25 0.258 0.268 0.278 0.287 0.294 0.298 0.302 0.305 0.306 0.306 0.302 0.297 0.293;0.142 0.144 0.147 0.152 0.156 0.161 0.165 0.17 0.174 0.178 0.183 0.188 0.193 0.199 0.205 0.211 0.217 0.223 0.23 0.237 0.245 0.254 0.263 0.271 0.279 0.285 0.289 0.292 0.296 0.301 0.305 0.306 0.306 0.305;0.137 0.139 0.142 0.147 0.152 0.157 0.162 0.167 0.172 0.177 0.182 0.187 0.192 0.198 0.203 0.209 0.214 0.22 0.226 0.233 0.241 0.249 0.257 0.265 0.271 0.276 0.279 0.282 0.287 0.293 0.299 0.304 0.306 0.307;0.135 0.137 0.14 0.145 0.15 0.156 0.161 0.166 0.172 0.178 0.183 0.188 0.194 0.199 0.204 0.209 0.213 0.218 0.224 0.231 0.238 0.246 0.254 0.261 0.266 0.27 0.272 0.275 0.279 0.284 0.291 0.296 0.299 0.301;0.136 0.138 0.141 0.145 0.151 0.157 0.163 0.168 0.174 0.18 0.186 0.192 0.197 0.202 0.207 0.211 0.215 0.219 0.224 0.23 0.237 0.244 0.252 0.259 0.263 0.266 0.268 0.27 0.273 0.277 0.282 0.286 0.289 0.29;0.137 0.139 0.142 0.147 0.153 0.159 0.165 0.171 0.177 0.183 0.189 0.195 0.2 0.205 0.21 0.214 0.217 0.22 0.224 0.229 0.236 0.243 0.251 0.258 0.263 0.265 0.267 0.268 0.27 0.272 0.275 0.277 0.279 0.28;0.138 0.14 0.143 0.148 0.154 0.16 0.166 0.173 0.179 0.185 0.191 0.197 0.202 0.207 0.212 0.216 0.219 0.221 0.225 0.229 0.236 0.243 0.251 0.257 0.262 0.265 0.267 0.267 0.268 0.27 0.271 0.272 0.273 0.274];

fc_extmp_map_cold=[NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN;105.6 107.6 111.9 118.5 127.9 140.3 155.7 174.3 196.2 220.9 247.6 274.5 300.1 322.9 342.3 358.6 372.7 385.3 397.3 408.8 419.8 429.7 438.2 444.8 449.6 452.6 454.3 455.1 455.4 455.5 455.6 455.6 455.6 455.6;123.6 126.8 133 142.3 154.4 169.4 186.8 206.3 227.3 249.3 271.4 292.7 312.6 330.5 346.4 360.5 373.4 385.6 397.3 408.8 419.7 429.7 438.2 444.8 449.5 452.5 454.2 455 455.3 455.4 455.4 455.3 455.3 455.3;146.3 150.4 158.3 169.3 183.1 199.1 216.8 235.6 254.8 273.7 291.5 308 323 336.9 349.8 362.1 374 385.6 397.2 408.5 419.3 429.1 437.4 444 448.6 451.5 453.1 453.8 453.9 453.8 453.6 453.5 453.4 453.3;163.3 167.8 176.1 187.6 201.4 216.8 233.5 250.8 268.2 285 300.6 314.9 327.9 340 351.6 363 374.1 385.1 395.9 406.4 416.4 425.6 433.4 439.5 443.9 446.5 447.8 448.1 447.9 447.4 446.9 446.4 446.1 446;173.5 177.9 186.3 197.6 211.1 225.9 241.7 258 274.4 290.1 304.9 318.4 330.9 342.4 353.3 363.8 373.6 383 391.8 400.1 407.8 414.9 421.1 426.2 430 432.5 433.8 434.1 433.8 433.2 432.4 431.8 431.3 431.1;181.7 186.2 194.5 205.8 219 233.6 248.8 264.4 280 295.1 309.3 322.5 334.6 345.6 355.6 364.6 372.5 379.1 384.6 389.1 393 396.6 400.3 403.9 407.3 410.2 412.3 413.7 414.5 414.7 414.6 414.4 414.1 414;189.4 193.9 202.2 213.5 226.5 240.7 255.4 270.4 285.2 299.7 313.4 326.2 337.9 348.4 357.6 365.2 371.1 375.1 377.3 378.1 378.3 378.7 379.8 382.1 385.3 389.1 392.8 396.2 398.8 400.8 402.1 402.9 403.3 403.5;195.5 199.8 207.8 218.6 231.1 244.7 258.7 273 287.1 301.1 314.4 327 338.6 348.9 357.7 364.8 369.9 372.8 373.6 372.7 371 369.4 369 370.3 373.4 377.7 382.8 387.8 392.4 396.2 399.2 401.3 402.7 403.3;201.3 205 211.9 221.3 232.5 244.9 257.9 271.2 284.6 297.9 311 323.5 335.2 346 355.4 363.2 369.1 372.7 374.1 373.4 371.6 369.5 368.2 368.6 371 375.3 380.9 387.1 393.2 398.8 403.6 407.3 409.9 411.3;207.9 210.5 215.7 223.1 232.3 243 254.7 266.9 279.3 292 304.6 317.2 329.4 340.9 351.5 360.7 368 373.3 376.2 376.9 375.7 373.7 371.9 371.3 372.9 376.7 382.4 389.5 396.9 404.2 410.6 415.9 419.6 421.6;213.4 215.1 218.7 224.3 231.9 241.2 251.7 262.9 274.6 286.6 299 311.6 324.1 336.4 347.8 358.1 366.8 373.4 377.6 379.4 379 377.2 375.2 374.2 375.3 379 385.1 392.9 401.3 409.6 417.1 423.2 427.6 429.9;215.8 217.1 219.9 224.6 231.4 240 250 260.9 272.2 283.9 296 308.4 320.9 333.3 345 355.7 364.9 372 376.9 379.3 379.5 378.2 376.7 376.3 378.2 382.9 390.1 399 408.4 417.5 425.4 431.7 436.1 438.4;215.6 216.8 219.4 224 230.6 239.2 249.3 260.2 271.5 283 294.7 306.6 318.6 330.5 341.9 352.4 361.5 368.8 373.8 376.6 377.3 377 376.9 378.2 382.3 389.4 399.1 410.2 421.5 431.8 440.3 446.7 450.9 453;214.3 215.4 218.1 222.7 229.5 238.3 248.6 259.7 271.1 282.5 293.8 305.1 316.3 327.5 338.3 348.4 357.3 364.4 369.5 372.5 373.9 374.7 376.2 379.9 386.8 397 409.9 424 437.8 449.9 459.6 466.5 470.9 472.9;213 214.2 216.9 221.6 228.4 237.4 247.9 259.3 270.8 282.1 293.1 303.8 314.5 325.1 335.5 345.3 354 361 366.1 369.4 371.2 372.9 375.8 381.3 390.3 402.9 418.3 434.7 450.6 464.4 475.2 482.8 487.5 489.7;211.9 213.1 215.8 220.6 227.5 236.6 247.2 258.7 270.3 281.6 292.6 303.2 313.8 324.3 334.6 344.3 353.1 360.2 365.6 369.1 371.5 373.9 377.7 384.3 394.6 408.4 424.9 442.3 458.9 473.3 484.4 492.3 497 499.2;210.1 211.3 214.1 218.8 225.9 235 245.8 257.4 269.3 280.8 292 303 313.8 324.6 335.3 345.4 354.6 362.3 368.2 372.6 376 379.5 384.6 392.5 403.9 418.6 435.5 452.9 469.1 482.9 493.4 500.6 505 506.9;206.8 208 210.8 215.7 222.8 232.2 243.2 255.1 267.3 279.4 291.2 302.8 314.3 325.9 337.2 348.1 358 366.5 373.4 379.1 384 389.4 396.4 406.1 419 434.7 451.9 469 484.5 497.2 506.6 512.9 516.5 518.1;203.1 204.4 207.3 212.4 219.8 229.4 240.7 253 265.6 278.1 290.5 302.8 315.1 327.5 339.6 351.2 361.7 371 379 385.8 392.3 399.3 408.1 419.5 433.9 450.5 468.1 485.1 500 511.9 520.5 526 529.1 530.5;200.8 202.3 205.8 211.5 219.6 229.9 241.8 254.3 267 279.6 292 304.5 317.2 329.9 342.4 354.3 365.2 374.9 383.4 391 398.4 406.7 416.8 429.3 444.5 461.6 479.3 496.1 510.6 522 530 535.1 537.9 539.1;200.1 202.3 207.1 214.6 224.7 236.7 249.6 262.6 274.9 286.8 298.5 310.4 322.6 335 347.3 359 369.8 379.5 388.2 396.4 404.7 414.1 425.4 439 454.7 471.8 489 505 518.5 529.1 536.5 541.1 543.6 544.7;199.8 203.1 209.9 220.2 233.4 248.3 263.3 277.1 289.2 300.1 310.4 320.9 331.9 343.3 354.8 365.9 376.2 385.8 394.7 403.6 413.2 424.2 437.1 452 468.2 485 501.2 515.8 527.9 537.2 543.5 547.4 549.5 550.3;199.7 204 212.7 225.6 241.8 259.6 276.8 291.8 304.2 314.5 323.6 332.7 342.2 352.2 362.5 372.6 382.2 391.4 400.5 410.1 421 433.8 448.6 464.9 481.8 498.5 513.8 527 537.7 545.6 550.9 554 555.6 556.3;203.2 208.1 217.8 232.2 250 269.4 288 304.2 317.4 327.9 336.7 344.8 352.9 361.2 369.5 377.4 385.1 392.6 400.7 410.4 422.6 437.5 454.8 473.3 491.7 508.8 523.7 536 545.6 552.4 556.8 559.3 560.5 561;220.3 225.3 235.2 249.4 266.9 285.8 304.3 320.9 334.8 346.2 355.4 363.1 369.5 374.9 379.1 382.2 384.4 387 391.6 400 413.6 432.2 454.5 477.8 499.9 518.9 534.3 546 554.5 560 563.4 565.1 565.8 566;259.7 264.2 272.9 285 299.5 315.4 331.4 346.8 360.6 372.7 382.5 389.7 394 395.2 393.2 388.3 381.8 375.9 374.3 380.1 395.5 419.8 449.9 481.1 509.4 532.1 548.9 560.2 567.3 571.5 573.5 574.3 574.4 574.3;312.3 315.3 321.2 329.2 338.8 349.7 361.6 374.2 387 399 409 415.7 418.1 415.6 407.8 395.4 380.4 366.4 358.5 361.6 378.4 407.9 445.3 484.2 518.7 545.4 563.7 574.9 581.1 584.1 585.3 585.4 585.2 585;354.7 356.4 359.5 363.8 369.4 376.5 385.3 395.8 407.6 419.3 429.3 435.9 437.5 433.3 423 407.3 388.4 370.7 359.8 361.2 378 409.1 449.3 491.3 528.2 556.4 575.1 586 591.5 593.8 594.5 594.5 594.2 594;380.4 382 385.2 389.5 395 401.9 410.6 421.1 432.7 444.4 454.3 460.8 462.6 458.9 449.7 435.7 419 403.2 393.4 394.6 409.7 437.8 474.3 512.3 545.9 571.1 587.6 596.8 601.1 602.6 602.7 602.4 601.9 601.7;403.3 407.3 414.3 423.1 432.8 442.8 453.3 464.4 475.6 486.1 494.7 500.4 502.3 500.1 494.1 485 474.3 464.5 459.1 461.6 474.1 495.9 523.5 551.9 576.6 594.9 606.5 612.5 614.9 615.2 614.6 613.8 613 612.6;430.1 437.6 450.6 466 481.3 494.9 506.7 517 526 533.6 539.5 543.1 544.4 543.4 540.4 536.1 531.5 528.1 527.9 532.7 543.2 558.8 577 594.9 609.8 620.3 626.6 629.6 630.3 629.9 628.9 627.9 627.1 626.7;447.4 457.1 473.9 493.4 512 527.5 539.4 548.5 555.4 560.6 564.3 566.5 567.1 566.4 564.8 562.8 561.3 561.3 564 570.2 580 592.5 605.9 618.1 627.7 634.1 637.7 639.1 639.1 638.5 637.6 636.7 636.1 635.7;NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN;NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN;NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN;NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN NaN];

fc_extmp_map_old=[101.5 103.3 106.9 112.8 121.4 132.9 147.6 165.8 187.8 213.2 241 269.5 296.6 320.8 341.2 358.1 372.5 385.3 397.3 408.8 419.8 429.7 438.2 444.8 449.6 452.6 454.3 455.1 455.4 455.5 455.6 455.6 455.6 455.6;105.6 107.6 111.9 118.5 127.9 140.3 155.7 174.3 196.2 220.9 247.6 274.5 300.1 322.9 342.3 358.6 372.7 385.3 397.3 408.8 419.8 429.7 438.2 444.8 449.6 452.6 454.3 455.1 455.4 455.5 455.6 455.6 455.6 455.6;113.3 115.8 121 128.9 139.7 153.3 169.8 189 210.6 234.1 258.7 283 305.9 326.4 344.2 359.5 373 385.4 397.3 408.8 419.8 429.7 438.2 444.8 449.6 452.6 454.3 455.1 455.4 455.5 455.5 455.5 455.5 455.5;123.6 126.8 133 142.3 154.4 169.4 186.8 206.3 227.3 249.3 271.4 292.7 312.6 330.5 346.4 360.5 373.4 385.6 397.3 408.8 419.7 429.7 438.2 444.8 449.5 452.5 454.2 455 455.3 455.4 455.4 455.3 455.3 455.3;135.2 138.8 146 156.4 169.6 185.4 203.2 222.6 242.8 263.1 282.8 301.4 318.5 334.1 348.3 361.4 373.7 385.6 397.3 408.7 419.6 429.5 438 444.6 449.3 452.3 453.9 454.6 454.9 454.9 454.9 454.8 454.8 454.7;146.3 150.4 158.3 169.3 183.1 199.1 216.8 235.6 254.8 273.7 291.5 308 323 336.9 349.8 362.1 374 385.6 397.2 408.5 419.3 429.1 437.4 444 448.6 451.5 453.1 453.8 453.9 453.8 453.6 453.5 453.4 453.3;155.9 160.2 168.4 179.8 193.7 209.6 226.8 244.9 263.1 280.7 297.2 312.2 326 338.7 350.8 362.6 374.1 385.5 396.8 407.8 418.4 427.9 436.1 442.5 447.1 449.9 451.3 451.8 451.8 451.5 451.1 450.9 450.7 450.6;163.3 167.8 176.1 187.6 201.4 216.8 233.5 250.8 268.2 285 300.6 314.9 327.9 340 351.6 363 374.1 385.1 395.9 406.4 416.4 425.6 433.4 439.5 443.9 446.5 447.8 448.1 447.9 447.4 446.9 446.4 446.1 446;168.9 173.4 181.8 193.2 206.8 221.9 238.1 254.8 271.6 287.8 302.9 316.7 329.4 341.1 352.4 363.3 374 384.3 394.3 403.9 413 421.3 428.5 434.2 438.3 440.8 442.1 442.3 441.9 441.3 440.6 439.9 439.5 439.3;173.5 177.9 186.3 197.6 211.1 225.9 241.7 258 274.4 290.1 304.9 318.4 330.9 342.4 353.3 363.8 373.6 383 391.8 400.1 407.8 414.9 421.1 426.2 430 432.5 433.8 434.1 433.8 433.2 432.4 431.8 431.3 431.1;177.6 182.1 190.4 201.7 215 229.7 245.2 261.2 277.1 292.5 307 320.4 332.6 343.9 354.4 364.2 373.1 381.2 388.5 395 400.9 406.4 411.4 415.8 419.3 421.9 423.5 424.2 424.3 423.9 423.4 422.9 422.5 422.3;181.7 186.2 194.5 205.8 219 233.6 248.8 264.4 280 295.1 309.3 322.5 334.6 345.6 355.6 364.6 372.5 379.1 384.6 389.1 393 396.6 400.3 403.9 407.3 410.2 412.3 413.7 414.5 414.7 414.6 414.4 414.1 414;185.7 190.2 198.5 209.8 223 237.4 252.4 267.6 282.8 297.6 311.6 324.6 336.5 347.2 356.7 365 371.8 377 380.7 383.2 385.1 387 389.3 392.2 395.4 398.7 401.7 404 405.6 406.7 407.2 407.4 407.5 407.5;189.4 193.9 202.2 213.5 226.5 240.7 255.4 270.4 285.2 299.7 313.4 326.2 337.9 348.4 357.6 365.2 371.1 375.1 377.3 378.1 378.3 378.7 379.8 382.1 385.3 389.1 392.8 396.2 398.8 400.8 402.1 402.9 403.3 403.5;192.6 197.1 205.3 216.4 229.3 243.2 257.6 272.2 286.7 300.9 314.5 327.1 338.7 349.1 357.9 365.1 370.4 373.6 374.9 374.5 373.5 372.7 373 374.7 377.9 382 386.5 390.8 394.5 397.4 399.5 400.9 401.8 402.2;195.5 199.8 207.8 218.6 231.1 244.7 258.7 273 287.1 301.1 314.4 327 338.6 348.9 357.7 364.8 369.9 372.8 373.6 372.7 371 369.4 369 370.3 373.4 377.7 382.8 387.8 392.4 396.2 399.2 401.3 402.7 403.3;198.3 202.3 209.9 220.1 232.1 245.2 258.8 272.6 286.4 300 313.2 325.8 337.4 347.8 356.9 364.2 369.5 372.6 373.4 372.5 370.5 368.6 367.7 368.5 371.3 375.7 381.1 386.8 392.2 396.9 400.7 403.6 405.6 406.6;201.3 205 211.9 221.3 232.5 244.9 257.9 271.2 284.6 297.9 311 323.5 335.2 346 355.4 363.2 369.1 372.7 374.1 373.4 371.6 369.5 368.2 368.6 371 375.3 380.9 387.1 393.2 398.8 403.6 407.3 409.9 411.3;204.6 207.7 213.8 222.2 232.5 244.1 256.4 269.2 282.1 295.1 307.9 320.5 332.4 343.6 353.5 362 368.6 373 375.1 375.1 373.6 371.4 369.9 369.8 371.7 375.8 381.5 388.1 394.9 401.4 407.1 411.6 414.8 416.5;207.9 210.5 215.7 223.1 232.3 243 254.7 266.9 279.3 292 304.6 317.2 329.4 340.9 351.5 360.7 368 373.3 376.2 376.9 375.7 373.7 371.9 371.3 372.9 376.7 382.4 389.5 396.9 404.2 410.6 415.9 419.6 421.6;210.9 213.1 217.4 223.8 232.1 242 253 264.7 276.7 289 301.5 314.1 326.5 338.5 349.5 359.3 367.4 373.4 377.1 378.4 377.6 375.7 373.7 372.9 374.1 377.8 383.6 391 399 406.9 413.9 419.7 423.9 426;213.4 215.1 218.7 224.3 231.9 241.2 251.7 262.9 274.6 286.6 299 311.6 324.1 336.4 347.8 358.1 366.8 373.4 377.6 379.4 379 377.2 375.2 374.2 375.3 379 385.1 392.9 401.3 409.6 417.1 423.2 427.6 429.9;215 216.4 219.6 224.6 231.7 240.5 250.7 261.7 273.1 284.9 297.2 309.7 322.3 334.7 346.4 356.9 365.9 372.9 377.5 379.7 379.6 378 376.2 375.3 376.6 380.6 387.1 395.4 404.3 413 420.7 427 431.4 433.7;215.8 217.1 219.9 224.6 231.4 240 250 260.9 272.2 283.9 296 308.4 320.9 333.3 345 355.7 364.9 372 376.9 379.3 379.5 378.2 376.7 376.3 378.2 382.9 390.1 399 408.4 417.5 425.4 431.7 436.1 438.4;215.9 217.1 219.8 224.4 231.1 239.6 249.6 260.5 271.8 283.4 295.2 307.4 319.7 331.9 343.6 354.2 363.4 370.6 375.6 378.2 378.7 377.9 376.9 377.3 380.1 385.8 394.1 403.9 414.2 423.8 431.9 438.3 442.5 444.7;215.6 216.8 219.4 224 230.6 239.2 249.3 260.2 271.5 283 294.7 306.6 318.6 330.5 341.9 352.4 361.5 368.8 373.8 376.6 377.3 377 376.9 378.2 382.3 389.4 399.1 410.2 421.5 431.8 440.3 446.7 450.9 453;215 216.2 218.8 223.4 230.1 238.8 248.9 260 271.3 282.8 294.3 305.8 317.4 329 340.1 350.4 359.4 366.6 371.7 374.5 375.6 375.9 376.6 379.1 384.6 393.3 404.6 417.2 429.7 440.9 449.9 456.5 460.7 462.8;214.3 215.4 218.1 222.7 229.5 238.3 248.6 259.7 271.1 282.5 293.8 305.1 316.3 327.5 338.3 348.4 357.3 364.4 369.5 372.5 373.9 374.7 376.2 379.9 386.8 397 409.9 424 437.8 449.9 459.6 466.5 470.9 472.9;213.6 214.8 217.5 222.1 228.9 237.8 248.3 259.5 271 282.3 293.4 304.4 315.3 326.1 336.7 346.6 355.4 362.5 367.5 370.6 372.3 373.6 375.9 380.6 388.7 400.3 414.5 430 444.9 458 468.3 475.6 480.1 482.2;213 214.2 216.9 221.6 228.4 237.4 247.9 259.3 270.8 282.1 293.1 303.8 314.5 325.1 335.5 345.3 354 361 366.1 369.4 371.2 372.9 375.8 381.3 390.3 402.9 418.3 434.7 450.6 464.4 475.2 482.8 487.5 489.7;212.4 213.7 216.4 221.1 228 237 247.6 259 270.6 281.9 292.8 303.5 314 324.5 334.8 344.5 353.2 360.3 365.4 368.8 370.9 372.9 376.2 382.3 392.1 405.4 421.4 438.6 455 469.2 480.4 488.2 492.9 495.1;211.9 213.1 215.8 220.6 227.5 236.6 247.2 258.7 270.3 281.6 292.6 303.2 313.8 324.3 334.6 344.3 353.1 360.2 365.6 369.1 371.5 373.9 377.7 384.3 394.6 408.4 424.9 442.3 458.9 473.3 484.4 492.3 497 499.2;211.2 212.4 215.1 219.9 226.9 236 246.7 258.2 269.9 281.3 292.3 303.1 313.7 324.3 334.7 344.6 353.5 360.9 366.5 370.4 373.2 376 380.4 387.6 398.4 412.7 429.4 446.8 463.4 477.5 488.5 496.1 500.7 502.8;210.1 211.3 214.1 218.8 225.9 235 245.8 257.4 269.3 280.8 292 303 313.8 324.6 335.3 345.4 354.6 362.3 368.2 372.6 376 379.5 384.6 392.5 403.9 418.6 435.5 452.9 469.1 482.9 493.4 500.6 505 506.9;208.6 209.8 212.6 217.4 224.5 233.7 244.6 256.4 268.4 280.2 291.6 302.9 314 325.2 336.2 346.6 356.1 364.2 370.6 375.6 379.7 384.1 390.1 398.9 411 426.2 443.2 460.5 476.3 489.5 499.5 506.3 510.3 512.1;206.8 208 210.8 215.7 222.8 232.2 243.2 255.1 267.3 279.4 291.2 302.8 314.3 325.9 337.2 348.1 358 366.5 373.4 379.1 384 389.4 396.4 406.1 419 434.7 451.9 469 484.5 497.2 506.6 512.9 516.5 518.1;204.9 206.1 209 213.9 221.1 230.6 241.8 253.9 266.3 278.6 290.7 302.7 314.7 326.6 338.4 349.6 359.9 368.8 376.3 382.6 388.4 394.6 402.6 413.2 426.9 443.1 460.5 477.6 492.7 504.9 513.9 519.7 523 524.5;203.1 204.4 207.3 212.4 219.8 229.4 240.7 253 265.6 278.1 290.5 302.8 315.1 327.5 339.6 351.2 361.7 371 379 385.8 392.3 399.3 408.1 419.5 433.9 450.5 468.1 485.1 500 511.9 520.5 526 529.1 530.5;201.7 203.1 206.2 211.5 219.1 229 240.6 253 265.7 278.3 290.8 303.3 315.9 328.5 340.9 352.7 363.5 373 381.3 388.6 395.6 403.3 412.7 424.8 439.6 456.6 474.3 491.2 506 517.6 525.9 531.2 534.1 535.4;200.8 202.3 205.8 211.5 219.6 229.9 241.8 254.3 267 279.6 292 304.5 317.2 329.9 342.4 354.3 365.2 374.9 383.4 391 398.4 406.7 416.8 429.3 444.5 461.6 479.3 496.1 510.6 522 530 535.1 537.9 539.1;200.3 202.1 206.1 212.5 221.5 232.5 244.8 257.4 270 282.3 294.5 306.8 319.3 332 344.5 356.4 367.3 377 385.6 393.5 401.4 410.1 420.8 433.8 449.3 466.4 484 500.4 514.6 525.6 533.4 538.3 540.9 542;200.1 202.3 207.1 214.6 224.7 236.7 249.6 262.6 274.9 286.8 298.5 310.4 322.6 335 347.3 359 369.8 379.5 388.2 396.4 404.7 414.1 425.4 439 454.7 471.8 489 505 518.5 529.1 536.5 541.1 543.6 544.7;199.9 202.7 208.4 217.3 228.8 242.2 256.1 269.3 281.6 292.9 304 315.2 326.9 338.9 350.8 362.3 372.9 382.5 391.3 399.8 408.7 418.9 431 445.1 461.1 478 494.8 510.1 523 532.9 539.8 544.1 546.4 547.4;199.8 203.1 209.9 220.2 233.4 248.3 263.3 277.1 289.2 300.1 310.4 320.9 331.9 343.3 354.8 365.9 376.2 385.8 394.7 403.6 413.2 424.2 437.1 452 468.2 485 501.2 515.8 527.9 537.2 543.5 547.4 549.5 550.3;199.6 203.5 211.3 223 237.9 254.3 270.4 284.8 297 307.5 317.1 326.9 337.1 347.9 358.8 369.4 379.5 388.9 398 407.3 417.5 429.4 443.3 458.8 475.4 492 507.8 521.6 533 541.5 547.3 550.8 552.6 553.4;199.7 204 212.7 225.6 241.8 259.6 276.8 291.8 304.2 314.5 323.6 332.7 342.2 352.2 362.5 372.6 382.2 391.4 400.5 410.1 421 433.8 448.6 464.9 481.8 498.5 513.8 527 537.7 545.6 550.9 554 555.6 556.3;200.5 205.2 214.5 228.3 245.5 264.3 282.4 298.1 310.8 321.1 329.9 338.5 347.3 356.5 365.9 375.2 384.1 392.7 401.6 411.4 422.9 436.7 452.5 469.7 487.2 504 519 531.7 541.8 549.1 554 556.8 558.2 558.8;203.2 208.1 217.8 232.2 250 269.4 288 304.2 317.4 327.9 336.7 344.8 352.9 361.2 369.5 377.4 385.1 392.6 400.7 410.4 422.6 437.5 454.8 473.3 491.7 508.8 523.7 536 545.6 552.4 556.8 559.3 560.5 561;209.3 214.3 224.2 238.7 256.7 276.1 294.9 311.5 325.1 335.9 344.9 352.8 360.1 367.1 373.7 379.7 385.1 390.7 397.4 406.7 419.5 436 455.3 475.8 495.7 513.5 528.6 540.6 549.6 555.9 559.8 561.9 562.9 563.2;220.3 225.3 235.2 249.4 266.9 285.8 304.3 320.9 334.8 346.2 355.4 363.1 369.5 374.9 379.1 382.2 384.4 387 391.6 400 413.6 432.2 454.5 477.8 499.9 518.9 534.3 546 554.5 560 563.4 565.1 565.8 566;237.2 242.1 251.5 265 281.4 299.1 316.7 332.9 347 358.7 368.3 375.7 381.1 384.5 385.8 385.1 383.2 381.9 383.5 390.8 405.2 426.6 452.5 479.5 504.5 525.2 541.2 552.7 560.4 565.3 568 569.2 569.6 569.6;259.7 264.2 272.9 285 299.5 315.4 331.4 346.8 360.6 372.7 382.5 389.7 394 395.2 393.2 388.3 381.8 375.9 374.3 380.1 395.5 419.8 449.9 481.1 509.4 532.1 548.9 560.2 567.3 571.5 573.5 574.3 574.4 574.3;285.8 289.7 297.1 307.3 319.5 332.9 347 361 374.5 386.6 396.5 403.4 406.7 405.8 400.7 391.7 380.6 370.3 365.3 369.7 385.9 413.1 447.2 482.6 514.2 539 556.6 567.9 574.5 578 579.5 579.9 579.8 579.6;312.3 315.3 321.2 329.2 338.8 349.7 361.6 374.2 387 399 409 415.7 418.1 415.6 407.8 395.4 380.4 366.4 358.5 361.6 378.4 407.9 445.3 484.2 518.7 545.4 563.7 574.9 581.1 584.1 585.3 585.4 585.2 585;335.8 338.1 342.4 348.3 355.6 364.2 374.2 385.6 397.8 409.6 419.6 426.2 428.1 424.4 414.8 400.1 382.4 365.9 356 358.1 375.1 405.9 445.5 486.7 523.1 550.9 569.7 580.9 586.8 589.4 590.3 590.3 590 589.8;354.7 356.4 359.5 363.8 369.4 376.5 385.3 395.8 407.6 419.3 429.3 435.9 437.5 433.3 423 407.3 388.4 370.7 359.8 361.2 378 409.1 449.3 491.3 528.2 556.4 575.1 586 591.5 593.8 594.5 594.5 594.2 594;368.9 370.4 373 376.8 381.7 388.2 396.6 406.8 418.5 430.2 440.2 446.8 448.4 444.3 434.1 418.7 400.2 382.8 371.9 373 389.2 419.4 458.5 499.4 535.4 562.8 580.7 591 596 598 598.4 598.3 597.9 597.7;380.4 382 385.2 389.5 395 401.9 410.6 421.1 432.7 444.4 454.3 460.8 462.6 458.9 449.7 435.7 419 403.2 393.4 394.6 409.7 437.8 474.3 512.3 545.9 571.1 587.6 596.8 601.1 602.6 602.7 602.4 601.9 601.7;391.2 393.8 398.4 404.5 411.7 419.9 429.4 440.3 451.9 463.2 472.7 478.9 480.8 477.9 470.2 458.4 444.4 431.3 423.4 425.1 438.9 464.1 496.5 530.3 559.9 582 596.3 604 607.4 608.3 608.1 607.5 606.9 606.5;403.3 407.3 414.3 423.1 432.8 442.8 453.3 464.4 475.6 486.1 494.7 500.4 502.3 500.1 494.1 485 474.3 464.5 459.1 461.6 474.1 495.9 523.5 551.9 576.6 594.9 606.5 612.5 614.9 615.2 614.6 613.8 613 612.6;416.7 422.4 432.4 444.6 457.1 469.1 480.4 491.2 501.5 510.8 518.1 522.8 524.5 523 518.7 512.2 504.7 498.3 495.7 499.3 510.7 529.1 551.7 574.5 594 608.3 617 621.4 622.9 622.7 621.9 620.9 620.1 619.7;430.1 437.6 450.6 466 481.3 494.9 506.7 517 526 533.6 539.5 543.1 544.4 543.4 540.4 536.1 531.5 528.1 527.9 532.7 543.2 558.8 577 594.9 609.8 620.3 626.6 629.6 630.3 629.9 628.9 627.9 627.1 626.7;441.1 450.1 465.5 483.5 500.9 515.8 527.7 537.3 545 551.1 555.7 558.4 559.3 558.5 556.4 553.6 551.1 549.9 551.6 557.3 567.3 580.8 595.9 610 621.5 629.4 633.8 635.8 636 635.5 634.5 633.6 632.9 632.5;447.4 457.1 473.9 493.4 512 527.5 539.4 548.5 555.4 560.6 564.3 566.5 567.1 566.4 564.8 562.8 561.3 561.3 564 570.2 580 592.5 605.9 618.1 627.7 634.1 637.7 639.1 639.1 638.5 637.6 636.7 636.1 635.7];

fc_hc_map_gpkWh=[9.762 8.875 4.406 2.874 2.092 1.623 1.363 1.232 1.181 1.155 1.122 1.078 1.035 1.005 0.9826 0.972 0.9588 0.9322 0.8981 0.8641 0.8368 0.8181 0.801 0.7828 0.7633 0.7556 0.7631 0.7936 0.8445 0.929 1.084 1.317 1.571 1.719;8.368 7.607 3.775 2.461 1.789 1.409 1.183 1.087 1.042 1.027 0.9974 0.9579 0.9204 0.8929 0.8734 0.864 0.8523 0.832 0.8014 0.7681 0.7438 0.7272 0.712 0.6982 0.6879 0.6965 0.7369 0.8077 0.9016 1.016 1.166 1.361 1.558 1.668;7.029 6.39 3.17 2.062 1.509 1.197 1.031 0.9563 0.9319 0.9185 0.8926 0.8621 0.8241 0.7997 0.7861 0.7776 0.7702 0.7547 0.7269 0.6993 0.677 0.6641 0.6524 0.6438 0.6487 0.6856 0.7705 0.9035 1.061 1.221 1.374 1.528 1.661 1.723;5.883 5.348 2.628 1.721 1.268 1.023 0.899 0.8496 0.8356 0.8247 0.8068 0.7754 0.7453 0.7199 0.7113 0.71 0.7088 0.7024 0.6864 0.6673 0.6478 0.6367 0.6308 0.6354 0.6608 0.7266 0.8547 1.04 1.251 1.444 1.597 1.709 1.781 1.799;4.974 4.522 2.219 1.451 1.078 0.879 0.7889 0.7486 0.7449 0.7372 0.7227 0.6954 0.6691 0.6469 0.643 0.6536 0.6735 0.6886 0.6903 0.6851 0.6762 0.6722 0.6743 0.6946 0.7396 0.83 0.9834 1.19 1.417 1.615 1.752 1.828 1.853 1.838;4.162 3.784 1.872 1.235 0.9265 0.7646 0.6892 0.6632 0.6583 0.6545 0.6398 0.6171 0.5917 0.5762 0.5796 0.6086 0.6559 0.7022 0.739 0.7576 0.7666 0.7728 0.7855 0.8175 0.8794 0.9815 1.131 1.321 1.521 1.692 1.801 1.845 1.84 1.806;3.506 3.188 1.576 1.05 0.806 0.6738 0.6158 0.5899 0.5796 0.5715 0.5615 0.5401 0.5192 0.5099 0.5227 0.5723 0.6475 0.733 0.807 0.8579 0.8893 0.909 0.9303 0.9686 1.032 1.126 1.25 1.397 1.546 1.67 1.744 1.763 1.744 1.704;2.938 2.671 1.335 0.9016 0.7015 0.6018 0.5466 0.5216 0.5114 0.5034 0.4902 0.4764 0.4592 0.4525 0.4757 0.5364 0.6339 0.7458 0.8509 0.9289 0.9805 1.011 1.037 1.072 1.126 1.197 1.283 1.377 1.471 1.547 1.589 1.593 1.569 1.53;2.475 2.25 1.141 0.7924 0.626 0.5452 0.4966 0.4754 0.4596 0.4473 0.4374 0.4265 0.4147 0.412 0.437 0.5008 0.6022 0.7197 0.8329 0.9225 0.9842 1.02 1.045 1.069 1.104 1.147 1.196 1.247 1.296 1.336 1.356 1.351 1.327 1.293;2.166 1.969 1.014 0.706 0.5743 0.5071 0.4674 0.4432 0.4288 0.4176 0.4087 0.3987 0.3903 0.3901 0.4112 0.4674 0.5519 0.6528 0.7524 0.8321 0.8875 0.9219 0.9424 0.9572 0.9733 0.9916 1.011 1.03 1.049 1.066 1.071 1.063 1.043 1.016;2.169 1.972 1 0.7044 0.5705 0.5071 0.4696 0.4468 0.4261 0.4132 0.4057 0.3996 0.3944 0.3944 0.4085 0.447 0.5071 0.5767 0.648 0.7058 0.7494 0.7768 0.7927 0.7999 0.803 0.8035 0.803 0.8014 0.802 0.8025 0.7992 0.7889 0.773 0.7522;2.525 2.295 1.174 0.8185 0.6539 0.5765 0.5338 0.5033 0.4771 0.4597 0.4484 0.4416 0.436 0.4353 0.4404 0.4609 0.4938 0.5338 0.5753 0.6125 0.6446 0.6686 0.6831 0.687 0.6829 0.6716 0.656 0.6406 0.6263 0.6148 0.6032 0.5907 0.5764 0.5605;3.375 3.068 1.559 1.073 0.8495 0.7353 0.6677 0.6231 0.5895 0.5635 0.5477 0.5348 0.5262 0.5188 0.518 0.5223 0.5325 0.5489 0.5705 0.5966 0.625 0.652 0.6696 0.6747 0.6646 0.6441 0.6183 0.5907 0.5642 0.5404 0.5198 0.5022 0.4873 0.4733;4.596 4.178 2.113 1.449 1.141 0.9805 0.8774 0.8073 0.7547 0.7164 0.6907 0.6718 0.654 0.6409 0.6296 0.6231 0.6188 0.6194 0.6292 0.652 0.687 0.7245 0.7497 0.756 0.7416 0.7139 0.679 0.6395 0.6003 0.5646 0.5337 0.5103 0.4913 0.4757;6.035 5.486 2.778 1.906 1.504 1.286 1.145 1.037 0.9595 0.899 0.8552 0.8236 0.7972 0.7731 0.7525 0.7346 0.7189 0.7092 0.7146 0.7413 0.7872 0.8353 0.8686 0.876 0.8587 0.8234 0.7802 0.7317 0.6817 0.6351 0.5947 0.5644 0.5417 0.5239;7.786 7.078 3.594 2.484 1.973 1.685 1.485 1.326 1.199 1.105 1.036 0.9862 0.9426 0.9023 0.8631 0.8305 0.802 0.7847 0.7889 0.8205 0.8732 0.9292 0.9672 0.975 0.9527 0.9128 0.8633 0.8093 0.7528 0.6995 0.6534 0.6181 0.5919 0.5719;10.46 9.509 4.839 3.346 2.652 2.24 1.948 1.706 1.511 1.369 1.268 1.189 1.113 1.035 0.9614 0.8988 0.8518 0.8266 0.83 0.8641 0.9202 0.977 1.014 1.017 0.9914 0.9483 0.8964 0.8413 0.7841 0.7301 0.6832 0.6462 0.6187 0.5974;14.64 13.31 6.755 4.625 3.611 2.994 2.553 2.197 1.92 1.724 1.586 1.468 1.342 1.202 1.065 0.9521 0.8773 0.8401 0.8418 0.8755 0.9301 0.9814 1.011 1.007 0.9771 0.9315 0.8816 0.8294 0.7774 0.7282 0.6843 0.6491 0.6225 0.6012;20.17 18.33 9.226 6.229 4.754 3.854 3.218 2.739 2.387 2.143 1.972 1.812 1.621 1.398 1.18 1.005 0.8911 0.8399 0.8366 0.8685 0.9158 0.9567 0.9753 0.9634 0.9297 0.8848 0.8372 0.791 0.7468 0.7056 0.6684 0.6374 0.6126 0.5922;25.54 23.22 11.61 7.732 5.78 4.587 3.769 3.185 2.778 2.509 2.312 2.116 1.869 1.576 1.285 1.053 0.9051 0.8375 0.8275 0.8531 0.8903 0.9195 0.9263 0.9073 0.8703 0.8264 0.7831 0.7437 0.7077 0.6742 0.6442 0.618 0.5958 0.5771;28.83 26.21 13.03 8.597 6.335 4.952 4.024 3.387 2.957 2.679 2.472 2.261 1.989 1.664 1.342 1.083 0.9203 0.8417 0.8221 0.8341 0.8576 0.8737 0.8702 0.8458 0.8082 0.7665 0.7273 0.6936 0.6643 0.6377 0.614 0.5924 0.5728 0.5549;28.72 26.11 12.96 8.511 6.234 4.844 3.917 3.29 2.868 2.594 2.391 2.186 1.928 1.625 1.327 1.088 0.9334 0.8538 0.8219 0.8173 0.8219 0.8227 0.81 0.7824 0.7447 0.7065 0.6712 0.6419 0.6171 0.5958 0.5759 0.5579 0.541 0.5246;25.61 23.28 11.56 7.601 5.578 4.344 3.527 2.968 2.584 2.325 2.132 1.949 1.736 1.493 1.257 1.069 0.9435 0.8691 0.8264 0.8025 0.7869 0.772 0.7507 0.7203 0.6846 0.6491 0.6177 0.591 0.5687 0.5497 0.5331 0.517 0.5019 0.4866;21.11 19.19 9.562 6.326 4.691 3.697 3.037 2.573 2.241 2.005 1.827 1.672 1.508 1.334 1.172 1.042 0.9488 0.8834 0.8325 0.7913 0.7582 0.7292 0.6997 0.6679 0.6332 0.6001 0.5701 0.5453 0.5241 0.506 0.4902 0.4755 0.4611 0.4472;17.03 15.48 7.75 5.182 3.899 3.128 2.61 2.239 1.959 1.75 1.59 1.454 1.329 1.209 1.103 1.016 0.9489 0.8912 0.8364 0.7849 0.7393 0.7003 0.6656 0.6312 0.5976 0.5648 0.5358 0.5101 0.489 0.4705 0.4548 0.4402 0.4264 0.413;14.39 13.08 6.577 4.436 3.385 2.76 2.338 2.026 1.786 1.598 1.449 1.329 1.224 1.135 1.059 0.9979 0.9451 0.8922 0.8375 0.7821 0.7307 0.6872 0.6489 0.614 0.5789 0.5459 0.5148 0.4889 0.4665 0.4483 0.4318 0.4164 0.4015 0.3884;13.16 11.96 6.026 4.087 3.14 2.577 2.2 1.92 1.699 1.523 1.383 1.268 1.173 1.097 1.035 0.9834 0.936 0.887 0.8336 0.7803 0.7294 0.6854 0.6468 0.6109 0.5755 0.5405 0.5089 0.4812 0.4581 0.4382 0.4201 0.4032 0.3873 0.3733;12.75 11.59 5.847 3.96 3.046 2.504 2.137 1.865 1.65 1.478 1.34 1.23 1.14 1.068 1.011 0.9631 0.9192 0.8745 0.8259 0.7763 0.7303 0.6893 0.654 0.6212 0.5868 0.5523 0.5194 0.4905 0.4647 0.4417 0.4197 0.3987 0.3804 0.3653;12.58 11.44 5.762 3.902 2.99 2.448 2.083 1.809 1.595 1.426 1.292 1.185 1.098 1.029 0.975 0.9316 0.8936 0.8543 0.8124 0.7689 0.729 0.697 0.6717 0.6486 0.6222 0.5922 0.5602 0.528 0.4966 0.4639 0.432 0.4026 0.3786 0.3611;12.35 11.23 5.654 3.815 2.913 2.374 2.006 1.733 1.521 1.355 1.225 1.121 1.039 0.9742 0.9242 0.8873 0.8566 0.8272 0.7934 0.7567 0.7264 0.7082 0.7021 0.6996 0.691 0.6716 0.6416 0.6046 0.5605 0.5114 0.4614 0.4174 0.3842 0.3626;12.01 10.92 5.492 3.697 2.809 2.274 1.911 1.642 1.435 1.274 1.149 1.051 0.972 0.9106 0.8646 0.8345 0.8149 0.7945 0.7696 0.741 0.7227 0.7238 0.7461 0.7746 0.7918 0.7874 0.7612 0.7167 0.6568 0.5844 0.5107 0.4469 0.4012 0.3745;11.63 10.57 5.312 3.567 2.698 2.174 1.818 1.555 1.352 1.198 1.078 0.9847 0.9092 0.8502 0.809 0.7837 0.7713 0.7603 0.7426 0.722 0.7165 0.74 0.7932 0.8548 0.9005 0.9128 0.8917 0.8409 0.7658 0.6735 0.5787 0.4975 0.4413 0.409;11.24 10.22 5.128 3.44 2.596 2.087 1.741 1.485 1.292 1.141 1.027 0.9359 0.8632 0.8046 0.7634 0.7404 0.7314 0.7234 0.7107 0.6973 0.7043 0.748 0.8287 0.9195 0.9889 1.015 0.9992 0.9462 0.8648 0.7633 0.6597 0.5713 0.5099 0.4749;10.88 9.895 4.966 3.327 2.511 2.019 1.684 1.438 1.252 1.109 0.997 0.9086 0.8349 0.7755 0.7315 0.7059 0.6934 0.6847 0.6734 0.6653 0.6828 0.7421 0.8422 0.9519 1.036 1.073 1.061 1.012 0.9348 0.8402 0.7442 0.6623 0.6038 0.5686;10.56 9.599 4.818 3.228 2.439 1.966 1.644 1.409 1.232 1.092 0.9829 0.8946 0.8201 0.7579 0.7098 0.6778 0.6581 0.6435 0.6312 0.6272 0.6532 0.7239 0.8353 0.9549 1.046 1.087 1.08 1.037 0.9729 0.896 0.8187 0.7521 0.7022 0.6692;10.27 9.341 4.682 3.141 2.376 1.92 1.612 1.387 1.215 1.081 0.9742 0.8856 0.8089 0.7448 0.6925 0.6534 0.6258 0.6056 0.5903 0.5897 0.6221 0.6998 0.817 0.9399 1.033 1.076 1.071 1.036 0.9848 0.9268 0.8696 0.8195 0.7791 0.7491;10 9.094 4.564 3.066 2.32 1.876 1.579 1.362 1.196 1.067 0.9613 0.8728 0.7972 0.7314 0.6767 0.6339 0.6015 0.577 0.5609 0.5611 0.597 0.6778 0.7963 0.9191 1.012 1.054 1.051 1.02 0.9776 0.9316 0.8871 0.847 0.8129 0.7848];

fc_nox_map_gpkWh=[14.64 13.31 6.593 4.311 3.265 2.929 3.201 3.948 4.937 5.91 6.726 7.423 8.141 9.06 10.25 11.66 13.15 14.51 15.64 16.45 16.9 16.97 16.72 16.25 15.7 15.18 14.76 14.45 14.24 14.18 14.4 14.96 15.61 15.89;11.84 10.76 5.325 3.512 2.691 2.479 2.78 3.486 4.381 5.253 5.973 6.593 7.236 8.049 9.108 10.36 11.68 12.89 13.9 14.61 15.02 15.09 14.87 14.46 14 13.61 13.37 13.27 13.27 13.36 13.6 14.04 14.5 14.66;9.037 8.216 4.082 2.722 2.143 2.069 2.409 3.094 3.924 4.716 5.365 5.92 6.496 7.225 8.168 9.294 10.47 11.56 12.47 13.13 13.51 13.59 13.42 13.08 12.73 12.5 12.49 12.68 12.97 13.25 13.52 13.81 14.04 14.04;6.796 6.178 3.089 2.09 1.717 1.752 2.128 2.773 3.533 4.252 4.836 5.335 5.847 6.493 7.33 8.329 9.379 10.37 11.22 11.87 12.28 12.41 12.32 12.08 11.87 11.82 12.03 12.46 12.97 13.39 13.66 13.79 13.81 13.66;5.393 4.902 2.472 1.719 1.479 1.564 1.93 2.524 3.196 3.827 4.34 4.776 5.219 5.777 6.493 7.353 8.273 9.181 10.03 10.75 11.28 11.55 11.58 11.49 11.43 11.55 11.9 12.45 13.02 13.46 13.66 13.66 13.51 13.24;4.72 4.291 2.185 1.56 1.385 1.49 1.82 2.324 2.887 3.416 3.854 4.22 4.584 5.032 5.612 6.312 7.093 7.937 8.834 9.72 10.46 10.95 11.17 11.25 11.35 11.59 11.99 12.49 12.96 13.28 13.37 13.25 12.99 12.67;4.383 3.985 2.065 1.509 1.386 1.5 1.775 2.173 2.617 3.039 3.391 3.682 3.96 4.294 4.719 5.248 5.882 6.672 7.645 8.736 9.753 10.51 10.95 11.2 11.43 11.72 12.08 12.44 12.72 12.85 12.81 12.6 12.3 11.96;4.202 3.82 1.995 1.51 1.428 1.555 1.792 2.101 2.426 2.739 3.009 3.23 3.426 3.646 3.924 4.285 4.776 5.491 6.497 7.742 8.995 10.01 10.68 11.1 11.42 11.72 11.99 12.19 12.28 12.24 12.09 11.84 11.53 11.2;4.01 3.645 1.933 1.521 1.49 1.642 1.865 2.101 2.33 2.553 2.748 2.91 3.04 3.17 3.321 3.541 3.893 4.495 5.443 6.711 8.078 9.288 10.17 10.75 11.13 11.41 11.6 11.69 11.67 11.54 11.33 11.07 10.77 10.46;3.938 3.58 1.924 1.551 1.566 1.754 1.979 2.173 2.338 2.486 2.628 2.744 2.827 2.889 2.953 3.065 3.287 3.739 4.524 5.665 7.009 8.323 9.381 10.12 10.57 10.84 10.98 11.02 10.97 10.82 10.6 10.35 10.07 9.786;4.029 3.663 1.986 1.625 1.669 1.882 2.108 2.286 2.405 2.508 2.609 2.697 2.752 2.77 2.777 2.812 2.928 3.223 3.802 4.742 5.98 7.328 8.519 9.388 9.908 10.18 10.31 10.35 10.31 10.17 9.976 9.738 9.483 9.218;4.433 4.03 2.175 1.762 1.795 2.013 2.238 2.402 2.499 2.58 2.664 2.74 2.776 2.772 2.745 2.731 2.769 2.936 3.342 4.11 5.244 6.594 7.862 8.815 9.38 9.664 9.802 9.858 9.834 9.72 9.536 9.311 9.068 8.817;5.076 4.615 2.472 1.961 1.952 2.135 2.35 2.507 2.605 2.688 2.774 2.845 2.874 2.858 2.813 2.769 2.761 2.855 3.172 3.862 4.957 6.305 7.592 8.557 9.127 9.419 9.57 9.644 9.63 9.518 9.331 9.105 8.864 8.617;5.924 5.385 2.85 2.214 2.125 2.265 2.451 2.605 2.714 2.817 2.92 2.997 3.023 2.998 2.946 2.89 2.859 2.928 3.241 3.949 5.074 6.427 7.686 8.604 9.143 9.432 9.598 9.684 9.666 9.538 9.331 9.089 8.839 8.588;6.77 6.155 3.239 2.474 2.305 2.397 2.547 2.69 2.815 2.945 3.064 3.15 3.173 3.147 3.094 3.031 2.995 3.077 3.44 4.222 5.413 6.771 7.972 8.807 9.293 9.572 9.753 9.842 9.808 9.651 9.417 9.154 8.889 8.629;7.567 6.879 3.616 2.734 2.508 2.549 2.661 2.781 2.905 3.045 3.177 3.265 3.289 3.262 3.205 3.143 3.117 3.236 3.676 4.555 5.81 7.156 8.277 9.017 9.447 9.716 9.901 9.983 9.923 9.735 9.472 9.187 8.907 8.641;8.368 7.607 4.015 3.036 2.758 2.743 2.803 2.886 2.985 3.118 3.246 3.333 3.351 3.321 3.265 3.208 3.204 3.382 3.916 4.892 6.188 7.493 8.518 9.163 9.543 9.799 9.977 10.04 9.945 9.725 9.438 9.136 8.847 8.576;9.327 8.479 4.514 3.415 3.073 2.998 2.995 3.02 3.078 3.189 3.307 3.386 3.398 3.361 3.302 3.252 3.28 3.532 4.17 5.237 6.551 7.791 8.704 9.26 9.595 9.835 9.996 10.02 9.895 9.645 9.339 9.028 8.735 8.465;10.6 9.636 5.14 3.875 3.443 3.292 3.225 3.19 3.216 3.305 3.411 3.477 3.478 3.434 3.372 3.33 3.391 3.715 4.445 5.581 6.89 8.052 8.858 9.334 9.631 9.851 9.984 9.973 9.808 9.536 9.222 8.909 8.616 8.348;12.13 11.03 5.86 4.383 3.841 3.614 3.49 3.424 3.435 3.517 3.612 3.663 3.65 3.596 3.53 3.499 3.588 3.965 4.754 5.921 7.2 8.282 8.999 9.415 9.679 9.871 9.965 9.91 9.712 9.424 9.108 8.798 8.51 8.246;13.93 12.66 6.701 4.957 4.288 3.985 3.825 3.752 3.77 3.852 3.936 3.967 3.935 3.872 3.81 3.793 3.909 4.319 5.128 6.277 7.496 8.497 9.147 9.525 9.759 9.909 9.946 9.839 9.61 9.312 8.997 8.694 8.412 8.151;16.02 14.57 7.668 5.631 4.822 4.452 4.267 4.197 4.228 4.306 4.37 4.37 4.319 4.251 4.199 4.207 4.362 4.802 5.608 6.702 7.829 8.741 9.337 9.685 9.887 9.979 9.944 9.775 9.51 9.2 8.887 8.589 8.313 8.057;18.39 16.72 8.79 6.424 5.469 5.031 4.818 4.736 4.754 4.81 4.841 4.809 4.739 4.671 4.64 4.692 4.907 5.398 6.205 7.227 8.237 9.04 9.573 9.889 10.05 10.07 9.955 9.719 9.416 9.094 8.781 8.487 8.215 7.963;20.8 18.91 9.922 7.236 6.143 5.634 5.382 5.27 5.257 5.275 5.266 5.207 5.126 5.065 5.066 5.177 5.482 6.057 6.886 7.834 8.71 9.382 9.832 10.1 10.21 10.16 9.963 9.665 9.328 8.992 8.677 8.384 8.114 7.863;22.94 20.86 10.93 7.956 6.743 6.171 5.872 5.714 5.654 5.628 5.59 5.517 5.437 5.391 5.428 5.615 6.036 6.722 7.596 8.481 9.219 9.744 10.09 10.29 10.34 10.21 9.948 9.602 9.238 8.893 8.574 8.277 8.003 7.751;24.66 22.42 11.73 8.529 7.215 6.584 6.244 6.042 5.938 5.879 5.823 5.749 5.68 5.653 5.73 6.003 6.55 7.359 8.287 9.124 9.732 10.11 10.34 10.45 10.42 10.23 9.906 9.527 9.149 8.797 8.47 8.162 7.877 7.62;26.27 23.88 12.47 9.039 7.626 6.939 6.555 6.314 6.171 6.086 6.02 5.951 5.892 5.884 6.001 6.354 7.015 7.938 8.915 9.715 10.22 10.47 10.58 10.58 10.45 10.19 9.832 9.438 9.058 8.703 8.368 8.046 7.745 7.48;28.23 25.66 13.35 9.631 8.096 7.346 6.919 6.636 6.455 6.342 6.259 6.188 6.133 6.138 6.29 6.709 7.456 8.452 9.449 10.21 10.62 10.77 10.75 10.63 10.41 10.09 9.701 9.306 8.935 8.59 8.256 7.928 7.617 7.347;30.93 28.12 14.57 10.47 8.759 7.92 7.429 7.096 6.869 6.713 6.603 6.511 6.447 6.456 6.633 7.099 7.895 8.909 9.879 10.57 10.89 10.93 10.8 10.56 10.24 9.86 9.458 9.078 8.734 8.419 8.111 7.8 7.499 7.233;34.35 31.22 16.17 11.59 9.653 8.682 8.105 7.705 7.422 7.219 7.063 6.937 6.85 6.854 7.05 7.543 8.351 9.331 10.22 10.81 11.02 10.93 10.67 10.31 9.897 9.471 9.068 8.716 8.418 8.157 7.905 7.644 7.378 7.131;37.96 34.51 17.94 12.88 10.69 9.555 8.864 8.387 8.042 7.785 7.582 7.414 7.298 7.289 7.491 7.993 8.784 9.694 10.47 10.93 11.01 10.79 10.39 9.911 9.409 8.945 8.549 8.235 7.995 7.805 7.63 7.44 7.228 7.01;41.08 37.35 19.57 14.1 11.69 10.39 9.581 9.024 8.623 8.321 8.074 7.867 7.721 7.695 7.89 8.377 9.121 9.938 10.59 10.92 10.88 10.55 10.03 9.442 8.867 8.376 7.99 7.712 7.523 7.398 7.294 7.171 7.011 6.823;43.25 39.32 20.79 15.08 12.5 11.07 10.16 9.541 9.094 8.754 8.474 8.239 8.071 8.027 8.202 8.642 9.303 10 10.53 10.75 10.64 10.24 9.651 8.992 8.371 7.864 7.488 7.233 7.074 6.984 6.919 6.834 6.707 6.541;44.42 40.39 21.55 15.73 13.07 11.57 10.6 9.931 9.45 9.082 8.777 8.523 8.345 8.292 8.438 8.804 9.336 9.877 10.27 10.41 10.27 9.867 9.28 8.612 7.981 7.472 7.103 6.855 6.698 6.604 6.534 6.45 6.329 6.173;44.85 40.78 21.93 16.11 13.43 11.9 10.91 10.21 9.703 9.311 8.99 8.724 8.548 8.497 8.617 8.894 9.27 9.623 9.866 9.933 9.803 9.461 8.932 8.304 7.693 7.195 6.83 6.578 6.403 6.279 6.175 6.065 5.934 5.78;44.77 40.7 22.01 16.24 13.6 12.07 11.08 10.37 9.842 9.431 9.097 8.828 8.661 8.62 8.723 8.921 9.148 9.327 9.429 9.439 9.337 9.069 8.621 8.051 7.475 6.993 6.634 6.374 6.177 6.017 5.875 5.736 5.589 5.434;44.23 40.21 21.81 16.14 13.54 12.05 11.06 10.35 9.819 9.401 9.063 8.795 8.637 8.607 8.701 8.851 8.988 9.059 9.075 9.052 8.969 8.755 8.365 7.84 7.292 6.826 6.473 6.211 6.002 5.823 5.659 5.502 5.347 5.193];

fc_pm_map_gpkWh=[0.1395 0.1268 0.09509 0.1057 0.1426 0.1775 0.2113 0.2264 0.2298 0.2324 0.2346 0.2305 0.2324 0.2536 0.2898 0.355 0.4358 0.5221 0.6057 0.6706 0.7195 0.7426 0.7434 0.7276 0.7026 0.6745 0.6437 0.6151 0.5864 0.5596 0.5304 0.4969 0.4636 0.438;0.124 0.1127 0.1127 0.1127 0.1409 0.169 0.1972 0.2093 0.2113 0.2066 0.2085 0.2049 0.2066 0.2254 0.2576 0.3156 0.3874 0.464 0.5353 0.5961 0.6396 0.6601 0.6608 0.6468 0.6222 0.595 0.5635 0.5343 0.5051 0.4799 0.4545 0.429 0.4015 0.3791;0.1674 0.1521 0.1268 0.1352 0.1648 0.1826 0.1859 0.1956 0.1902 0.1859 0.1876 0.1844 0.1859 0.2029 0.2318 0.284 0.3487 0.4176 0.4818 0.5338 0.5731 0.5917 0.5924 0.5777 0.5557 0.5254 0.4935 0.4583 0.4293 0.404 0.3837 0.3632 0.3439 0.3273;0.2536 0.2305 0.1614 0.169 0.1844 0.1844 0.1844 0.1778 0.1729 0.1742 0.1706 0.1718 0.1729 0.1844 0.2141 0.2582 0.317 0.377 0.4329 0.4804 0.5117 0.5291 0.5281 0.5131 0.4898 0.4592 0.4238 0.3876 0.3573 0.3354 0.3212 0.3079 0.2939 0.2822;0.3254 0.2958 0.2113 0.1972 0.1902 0.1859 0.1761 0.169 0.1585 0.1597 0.1564 0.1614 0.162 0.1723 0.1992 0.2367 0.2853 0.3356 0.3827 0.4204 0.448 0.4588 0.4572 0.441 0.4173 0.3871 0.3527 0.3193 0.2943 0.2783 0.2691 0.2617 0.2549 0.2459;0.4291 0.3901 0.2341 0.2081 0.1951 0.1794 0.1625 0.156 0.1463 0.1474 0.1482 0.1525 0.156 0.168 0.1867 0.2185 0.256 0.2937 0.3294 0.3573 0.3745 0.3808 0.3759 0.3596 0.3381 0.3105 0.2821 0.2572 0.2424 0.2341 0.2328 0.2303 0.2267 0.2211;0.518 0.4709 0.2717 0.2173 0.1902 0.1666 0.1509 0.1397 0.1358 0.1368 0.1449 0.1515 0.157 0.1644 0.1811 0.2029 0.2309 0.2578 0.2797 0.2955 0.3025 0.3001 0.2898 0.2756 0.2581 0.2376 0.2201 0.208 0.2031 0.2061 0.2113 0.2138 0.2117 0.2075;0.595 0.5409 0.2874 0.2141 0.1775 0.1555 0.1352 0.1256 0.1268 0.1315 0.142 0.1506 0.1578 0.1664 0.1811 0.2006 0.2198 0.2387 0.2517 0.2545 0.2502 0.2367 0.2228 0.2073 0.193 0.1826 0.1768 0.1778 0.1847 0.1959 0.2062 0.2127 0.2134 0.209;0.6624 0.6022 0.3011 0.2113 0.1743 0.1458 0.1268 0.1223 0.1228 0.1303 0.1426 0.1556 0.1638 0.1755 0.1902 0.2092 0.2298 0.2461 0.2536 0.2502 0.2346 0.2113 0.1873 0.1695 0.1585 0.1547 0.1597 0.1714 0.1868 0.2044 0.2187 0.228 0.2298 0.2276;0.6891 0.6265 0.3132 0.2188 0.1715 0.1432 0.1293 0.1193 0.1231 0.1326 0.1462 0.1573 0.1715 0.1882 0.2067 0.2307 0.2573 0.2773 0.2884 0.2826 0.2595 0.2259 0.1912 0.166 0.1529 0.1551 0.1675 0.1856 0.2078 0.2284 0.2456 0.2569 0.262 0.2603;0.7128 0.648 0.324 0.216 0.169 0.1465 0.1315 0.1288 0.1303 0.1377 0.1521 0.1665 0.1831 0.2037 0.2294 0.2611 0.2941 0.3232 0.3397 0.3336 0.3071 0.2643 0.219 0.185 0.169 0.1713 0.1886 0.2129 0.2385 0.2623 0.2817 0.2954 0.302 0.3005;0.7047 0.6406 0.3336 0.2313 0.1802 0.1601 0.1468 0.1411 0.1435 0.1542 0.1708 0.1868 0.2046 0.2279 0.2574 0.2954 0.3353 0.3705 0.39 0.3863 0.3563 0.3063 0.2524 0.2124 0.1935 0.1964 0.2176 0.2452 0.2755 0.3028 0.3247 0.3401 0.3478 0.3462;0.7531 0.6846 0.3423 0.2367 0.1965 0.1724 0.1648 0.1666 0.1712 0.1859 0.2029 0.2213 0.2388 0.2614 0.2916 0.3296 0.374 0.4117 0.4325 0.4271 0.3918 0.3381 0.2812 0.2381 0.2198 0.2252 0.2477 0.2808 0.3161 0.348 0.3736 0.3902 0.397 0.3942;0.7704 0.7003 0.3622 0.2576 0.2113 0.198 0.1932 0.1966 0.2083 0.2254 0.2463 0.2656 0.2838 0.3028 0.3312 0.3671 0.409 0.4475 0.4655 0.455 0.4166 0.3588 0.3019 0.2614 0.2455 0.2531 0.2805 0.3193 0.3605 0.3989 0.4282 0.4456 0.4505 0.4457;0.8114 0.7377 0.3804 0.2766 0.2363 0.2259 0.2228 0.2338 0.2478 0.2715 0.2951 0.3164 0.3304 0.3458 0.3672 0.4011 0.4409 0.4746 0.4892 0.4732 0.4288 0.371 0.3164 0.2796 0.2689 0.2812 0.3121 0.3552 0.4034 0.4475 0.481 0.4975 0.4999 0.4925;0.8489 0.7717 0.4079 0.3013 0.2701 0.2646 0.2646 0.274 0.2894 0.3136 0.3396 0.3588 0.3712 0.3816 0.4 0.4292 0.4644 0.4955 0.5035 0.4828 0.4355 0.3769 0.3257 0.2953 0.2894 0.306 0.3401 0.3871 0.4402 0.4881 0.5226 0.5384 0.5381 0.5278;0.8833 0.803 0.4332 0.3381 0.317 0.317 0.3205 0.326 0.3354 0.3569 0.3782 0.3957 0.4068 0.4129 0.4256 0.4508 0.4834 0.5084 0.5118 0.4849 0.4353 0.3773 0.3304 0.3059 0.3064 0.3271 0.3657 0.4148 0.4702 0.5188 0.5515 0.5651 0.5619 0.5494;0.9372 0.852 0.4767 0.3922 0.3804 0.3854 0.3888 0.3854 0.3905 0.4035 0.4199 0.4352 0.4412 0.4447 0.4535 0.4733 0.5008 0.5191 0.5162 0.4847 0.4321 0.3758 0.3338 0.3157 0.322 0.3481 0.3901 0.4418 0.497 0.5435 0.5727 0.5824 0.5762 0.5618;1.008 0.9167 0.5266 0.4486 0.4486 0.4603 0.4649 0.457 0.4535 0.4595 0.472 0.4805 0.4828 0.4816 0.4848 0.5006 0.5193 0.5312 0.5212 0.4835 0.4291 0.3743 0.3378 0.3257 0.3381 0.3698 0.4156 0.4703 0.5246 0.5677 0.593 0.5977 0.5882 0.5727;1.116 1.014 0.601 0.5134 0.5165 0.5334 0.5384 0.5286 0.5212 0.5217 0.5278 0.531 0.5259 0.5187 0.5179 0.5247 0.5365 0.5392 0.5217 0.4804 0.4254 0.3739 0.3432 0.3381 0.3569 0.3937 0.4436 0.4995 0.5534 0.5939 0.6136 0.6138 0.6016 0.5845;1.335 1.214 0.6973 0.5977 0.5932 0.6086 0.6098 0.6003 0.5909 0.5836 0.5814 0.5763 0.5645 0.5503 0.5395 0.5373 0.5388 0.5338 0.5122 0.4709 0.4193 0.3743 0.3515 0.3536 0.3781 0.4195 0.473 0.5313 0.5835 0.6202 0.6351 0.631 0.6164 0.5988;1.654 1.504 0.8569 0.7053 0.6864 0.6925 0.6908 0.677 0.658 0.6431 0.6295 0.6121 0.5902 0.5663 0.5459 0.5316 0.5235 0.5112 0.4877 0.4501 0.4075 0.3747 0.3641 0.3756 0.4073 0.4526 0.5078 0.5661 0.6158 0.6476 0.6569 0.6493 0.6328 0.6136;2.12 1.927 1.073 0.8565 0.8072 0.7979 0.7861 0.7583 0.7269 0.6968 0.6677 0.6378 0.6057 0.5735 0.5422 0.5173 0.4966 0.4773 0.4545 0.4244 0.3947 0.3783 0.3827 0.4072 0.4473 0.497 0.5539 0.6104 0.655 0.6803 0.6829 0.6707 0.6514 0.6311;2.735 2.487 1.358 1.052 0.957 0.9194 0.8888 0.846 0.7955 0.7453 0.7018 0.6588 0.6176 0.5776 0.5387 0.5039 0.4744 0.4494 0.4262 0.4038 0.3893 0.3911 0.4142 0.4552 0.5078 0.5654 0.6235 0.675 0.711 0.7249 0.7182 0.6982 0.6738 0.6509;3.574 3.249 1.727 1.3 1.137 1.059 0.9984 0.9305 0.8617 0.7942 0.7338 0.6815 0.6339 0.59 0.5479 0.5082 0.4725 0.4428 0.4191 0.4029 0.4017 0.4219 0.4661 0.5292 0.6002 0.6701 0.7314 0.7766 0.7986 0.7968 0.7744 0.7403 0.7037 0.6733;4.683 4.257 2.221 1.608 1.349 1.211 1.112 1.014 0.9221 0.8401 0.7684 0.7083 0.6583 0.6135 0.5708 0.5307 0.4937 0.4619 0.438 0.4271 0.4372 0.4757 0.5435 0.6327 0.7293 0.8169 0.8842 0.9221 0.9281 0.9056 0.8606 0.8026 0.7453 0.7027;6.005 5.459 2.804 1.969 1.589 1.372 1.221 1.091 0.9788 0.88 0.7995 0.7363 0.6849 0.6414 0.602 0.5648 0.5295 0.4984 0.4765 0.4702 0.4922 0.5483 0.64 0.7562 0.8794 0.9886 1.065 1.098 1.09 1.046 0.975 0.8882 0.8041 0.7444;7.236 6.578 3.347 2.299 1.8 1.507 1.304 1.145 1.012 0.9032 0.8187 0.7535 0.704 0.6643 0.6303 0.5979 0.566 0.5387 0.5208 0.5224 0.5557 0.6279 0.7377 0.875 1.02 1.146 1.231 1.263 1.246 1.187 1.096 0.9862 0.8798 0.8053;8.042 7.311 3.698 2.508 1.919 1.569 1.331 1.153 1.013 0.9 0.8142 0.7517 0.7067 0.6729 0.645 0.6189 0.5934 0.5718 0.5611 0.5739 0.6191 0.7037 0.8241 0.9696 1.122 1.253 1.34 1.371 1.352 1.291 1.197 1.081 0.9672 0.8871;8.217 7.47 3.762 2.527 1.909 1.538 1.288 1.108 0.9697 0.865 0.7854 0.7302 0.6933 0.6674 0.6471 0.6278 0.6091 0.595 0.5955 0.6211 0.6792 0.7708 0.8922 1.032 1.174 1.293 1.371 1.398 1.382 1.332 1.253 1.154 1.054 0.9798;7.927 7.207 3.61 2.407 1.798 1.433 1.194 1.026 0.9042 0.8111 0.7447 0.6988 0.6706 0.6539 0.6415 0.629 0.6164 0.6115 0.6235 0.6631 0.7327 0.8287 0.9439 1.068 1.186 1.279 1.336 1.355 1.345 1.312 1.262 1.194 1.121 1.062;7.467 6.788 3.381 2.241 1.661 1.316 1.092 0.9437 0.8387 0.7629 0.7087 0.675 0.6556 0.6462 0.6381 0.6294 0.6217 0.6234 0.6458 0.6967 0.7744 0.8725 0.9806 1.085 1.175 1.237 1.269 1.276 1.268 1.253 1.23 1.194 1.15 1.107;7.14 6.491 3.22 2.113 1.559 1.232 1.025 0.8893 0.7987 0.7368 0.6948 0.6696 0.6582 0.6524 0.6475 0.6381 0.63 0.6339 0.6607 0.7167 0.7994 0.8971 0.9981 1.088 1.154 1.19 1.2 1.194 1.185 1.179 1.172 1.157 1.132 1.1;7.048 6.407 3.16 2.07 1.521 1.2 1.002 0.8747 0.7916 0.738 0.7038 0.6871 0.6813 0.6784 0.6715 0.6589 0.6455 0.6447 0.6686 0.7226 0.8034 0.8994 0.9968 1.077 1.129 1.148 1.143 1.127 1.113 1.106 1.102 1.093 1.075 1.049;7.106 6.46 3.182 2.081 1.53 1.21 1.012 0.8866 0.8075 0.758 0.7293 0.7168 0.7154 0.7133 0.7046 0.6858 0.6649 0.6563 0.6715 0.7169 0.7921 0.8849 0.9797 1.057 1.103 1.115 1.102 1.078 1.057 1.043 1.032 1.019 0.9996 0.9748;7.187 6.534 3.22 2.103 1.551 1.229 1.03 0.9048 0.8256 0.7771 0.7524 0.7441 0.745 0.7439 0.7329 0.71 0.6826 0.6646 0.6696 0.7051 0.7725 0.8615 0.9553 1.032 1.077 1.086 1.07 1.043 1.016 0.9927 0.973 0.953 0.9302 0.9056;7.176 6.524 3.216 2.105 1.553 1.233 1.033 0.9089 0.8299 0.7825 0.7596 0.7534 0.7549 0.7545 0.7426 0.7177 0.6865 0.6638 0.6621 0.6909 0.7526 0.8386 0.931 1.007 1.052 1.061 1.044 1.015 0.9846 0.957 0.9321 0.9076 0.8832 0.8581];

% End added by ADVISOR 2002 converter: 11-Apr-2003