%
% This is a MATLAB example of performing DOE for a box with rounded
%	corneres
%
%	4 design variables and 3 responses.
%
%	Design variables:	Input_1 = L0 - Box Width
%						Input_2 = L1 - Box Height
%						Input_3 = L2 - Box Depth
%						Input_4 = R	 - Filet Radius
%
%	Responses:	response_1 = V - Box Volume
%				response_2 = TEST   - CAD Flag
%				response_3 = S   - Box Surface
%
%	Optimization Problem:
%				Minimize a volume of the box, sibjected to surface area
%				constraints and feasibility (pass/fail) constraint from
%				CAD.
%
%			Pi = 3.14
%
%			V    = (L0*L2 - 4*R*R + Pi*R*R)*L1
%			S    = 2*(L0*L2 + (Pi-4)*R*R) + L1*(2*L0 + 2*L2 +2*(Pi-4)*R)
%
%			TEST = TRUE (-1)  (L0-2*R >= 0)  AND  (L2-2*R >= 0)
%			       FALSE (1)  (L0-2*R < 0)  OR  (L2-2*R < 0)
%
%			If TEST==FALSE, then we consider that neither V or S could
%				be evaluated and assign the value of 0 to both V and S.
%
%			6000 <= S <= 10000
%
%	Starting point:	Input_1 = L0 = 30
%					Input_2 = L1 = 30
%					Input_3 = L2 = 30
%					Input_4 = R	 = 10


% Clear all the variables
clear all

% Set up constant parameters
pi = 3.14159265;
NInputs = 4;
NResps = 3;

% Set up all input parameters: database IDs, initial values, lower and upper bounds
vDBIDInp=[0,0,0,0];
vX = [30.0,30.0,30.0,10.0];
vLBInp = [30.0,30.0,30.0,10.0];
vUBInp = [50.0,50.0,50.0,20.0];

% Set up all response parameters: database IDs,lower and upper bounds
vDBIDResp = [0,0,0];
vLBResp = [-1.e+30,-1.e+30, 6000.0];
vUBResp = [ 1.e+30, 0,     10000.0];

% Open the database
    disp(' ');
    disp(' Before openning the database');
    
Error = VDOC_OpenDatabase( 'box_discr' );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_OpenDatabase.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Check if the database is really open
Error = VDOC_IsOpen;
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_IsOpen.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end


%	Put input parameters into the database
[Error,vDBIDInp] = VDOC_PutInputAll( vDBIDInp, vX, vLBInp, vUBInp);
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutInputAll.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Put just the bounds for all the responses into the database
[Error,vDBIDResp] = VDOC_PutRespAll( vDBIDResp, vLBResp, vUBResp );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutRespAll.  Error =');
    Error
 	Error = VDOC_CloseDatabase;
   return;
end

% Change the type of the 2nd response (TEST) to be pass/fail
TypeRespPassFail = 3;
Error = VDOC_ChangeRespType(  vDBIDResp(2), TypeRespPassFail );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_ChangeRespType  (Pass/Fail).  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Set the task type information 
%	2 - DOE
TaskType = 2 
Error = VDOC_PutTaskType( TaskType , 'Rounded box with discrete variables. DOE.' );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutTaskType.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Change the action to be performed by DOE to construct the model 
%	2 - DOE
DOEAction = 2 
Error = VDOC_PutDOEControlGeneral( DOEAction );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutDOEControlGeneral.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Set the design to be Hybrid #3 for 4 variables 
%	2 - DOE
Error = VDOC_PutDOEControlDesign( 7, 4163 )
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutDOEControlDesign.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Make the task
TaskNumber = VDOC_MakeTask;
if TaskNumber < 0 
    disp(' ');
    disp(' Stopped after VDOC_MakeTask.  Error =');
    TaskNumber 
 	Error = VDOC_CloseDatabase;
    return;
end

% Set up the loop with DOE
%	StatusCode is set to initialize the process
StatusCode = 0;
matDVs = 0;
matResps = 0;
icount = 0;

% Loop around call to DOE and evaluation of responses
%	The loop continues for the cases of:
%	StatusCode = 0	- Initialization 
%	StatusCode = 2	- evaluate response at one point
%	StatusCode = 3	- evaluate response at several points (for gradients)
while( (StatusCode==0) | (StatusCode==2) | (StatusCode==3) )
 
    % Call Optimizer
    [Error, StatusCode, matDVs, matResps] = VDOC_RunDOE( TaskNumber, StatusCode, matDVs, matResps );
 	if Error < 0 
    	disp(' ');
    	disp(' Stopped after VDOC_RunRSA.  Error =');
    	Error
    	return;
	end
   
    % If the process is completed (StatusCode=1), or if some unexpected 
    %	error occured - exit.
    if ( (StatusCode==1) & (StatusCode==6) )
        StatusCode
        break;
    end
    
    
    % Check if we need to evaluate responses:
	%				1st response: V
	%				2nd response: TEST 
	%				3rd response: S  
    if ( StatusCode == 2 )
    
    	% Get the number of points where we need to evaluate responses:
    	%	number of rows (first dimension) in the matrix of design variables
    	NSize = size( matDVs );
    	NPoints = NSize(1);
    
    	% For each point evaluate responses and write them into the matrix of responses
   		for i = 1:NPoints
        	% Extract design variables
        	L0 = matDVs(i,1);
        	L1 = matDVs(i,2);
        	L2 = matDVs(i,3);
        	R  = matDVs(i,4);
        
            % Evaluate responses
            V = L1*( L0*L2 - 4*R*R + pi*R*R );
			S = 2*( L0*L2 + (pi-4)*R*R ) + L1*( 2*L0 + 2*L2 + 2*(pi-4)*R );

  			if( ((L0-2*R)>=0) & ((L2-2*R)>=0) )
				TEST = -1;
			else
                V    = 0;
                TEST = 1;
				S    = 0;   
			end          
        
        	% Put responses into the corresponding rows of the matrix of responses
        	matResps(i,1) = V;
        	matResps(i,2) = TEST;
        	matResps(i,3) = S;
        end 
        
        disp(' ');
        disp(' Counting ');
        icount
        disp(' Status Code');
        StatusCode
		disp(' ====================================');
    	matDVs
   		disp(' -----');
     	matResps    
	end
end  

disp(' ');
disp(' ====================================');
disp(' The run is completed!!!');
StatusCode
disp(' ====================================');

% Get all the points
[Error, matDVs, matResps, vToAnalyze] = VDOC_GetDPointsAll( TaskNumber )
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_GetDPointsAll.  Error =');
    Error
    return;
end

% Get efficiencies of the design
[Error, DEffic, AEffic, GEffic, ScVar, LnDet, Resol] = VDOC_GetDOEEffic ( TaskNumber )
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_GetDOEEffic.  Error =');
    Error
    return;
end


% Get ANOVA characteristics for each response
for i = 1:NResps 
    disp( 'Response #' );
    i
	[Error, vParamsANOVA] = VDOC_GetDOEANOVA( TaskNumber, vDBIDResp(i) )
	if Error < 0 
    	disp(' ');
	    disp(' Stopped after VDOC_GetDOEANOVA.  Error =');
	    Error
	    return;
	end
end    


% Get residuals for each response
disp( ' ' );
disp( ' ################## Residuals ############################' );
disp( ' ' );
for i = 1:NResps 
    disp( 'Response #' );
    i
	[Error, vPredResp, vResi, vStdResi, vStuResi, vRStuResi,vPRESSResi, vCookD, vHDiag] = VDOC_GetDOEResiduals(TaskNumber, vDBIDResp(i))

	if Error < 0 
    	disp(' ');
	    disp(' Stopped after VDOC_GetDOEResiduals.  Error =');
	    Error
	    return;
	end
end    


% Get models for each response
disp( ' ' );
disp( ' !!!!!!!!!!!!!!!!!!!! Models !!!!!!!!!!!!!!!!!!!!' );
disp( ' ' );
for i = 1:NResps 
    disp( 'Response #' );
    i
	[Error, vCoeffs, vModelExp, vStdErr, vTStat, vPVal] = VDOC_GetDOEModel(TaskNumber, vDBIDResp(i))

	if Error < 0 
    	disp(' ');
	    disp(' Stopped after VDOC_GetDOEResiduals.  Error =');
	    Error
	    return;
	end
end    


% Close the database
Error = VDOC_CloseDatabase;
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_CloseDatabase.  Error =');
    Error
    return;
end



