%
% This is a MATLAB example of optimizing a box with rounded corneres
%
%	4 design variables and 3 responses.
%
%	Design variables:	Input_1 = L0 - Box Width
%						Input_2 = L1 - Box Height
%						Input_3 = L2 - Box Depth
%						Input_4 = R	 - Filet Radius
%
%	Responses:	response_1 = V - Box Volume
%				response_2 = TEST   - CAD Flag
%				response_3 = S   - Box Surface
%
%	Problem:  Minimize a volume of the box, sibjected to surface area
%				constraints and feasibility (pass/fail) constraint from
%				CAD.
%
%			Pi = 3.14
%
%			V    = (L0*L2 - 4*R*R + Pi*R*R)*L1
%			S    = 2*(L0*L2 + (Pi-4)*R*R) + L1*(2*L0 + 2*L2 +2*(Pi-4)*R)
%
%			TEST = TRUE (-1)  (L0-2*R >= 0)  AND  (L2-2*R >= 0)
%			       FALSE (1)  (L0-2*R < 0)  OR  (L2-2*R < 0)
%
%			If TEST==FALSE, then we consider that neither V or S could
%				be evaluated and assign the value of 0 to both V and S.
%
%			6000 <= S <= 10000
%
%	Starting point:	Input_1 = L0 = 30
%					Input_2 = L1 = 30
%					Input_3 = L2 = 30
%					Input_4 = R	 = 10


% Clear all the variables
clear all

% Set up constant parameters
pi = 3.14159265;
NInputs = 4;
NResps = 3;

% Set up all input parameters: database IDs, initial values, lower and upper bounds
vDBIDInp=[0,0,0,0];
vX = [30.0,30.0,30.0,10.0];
vLBInp = [30.0,30.0,30.0,10.0];
vUBInp = [50.0,50.0,50.0,20.0];

% Set up all response parameters: database IDs,lower and upper bounds
vDBIDResp = [0,0,0];
vLBResp = [-1.e+30,-1.e+30, 6000.0];
vUBResp = [ 1.e+30, 0,     10000.0];

% Create two discrete sets for the input parameters
vSetL = [30,35,40,45,50];
vSetR = [10,15,20];


% Open the database
    disp(' ');
    disp(' Before openning the database');
    
Error = VDOC_OpenDatabase( 'box_discr' );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_OpenDatabase.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Check if the database is really open
Error = VDOC_IsOpen;
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_IsOpen.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Put two discrete sets into the database
SetIDs = [0,0];

[Error,SetIDs(1)] = VDOC_PutDiscreteSet( SetIDs(1), vSetL, 'Set_L' );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutDiscreteSet (1st set).  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

[Error,SetIDs(2)] = VDOC_PutDiscreteSet( SetIDs(2), vSetR, 'Set_R' );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutDiscreteSet (2nd set).  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

%	Put input parameters into the database
[Error,vDBIDInp] = VDOC_PutInputAll( vDBIDInp, vX, vLBInp, vUBInp);
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutInputAll.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Change the type of the first three inputs to be discrete	
%	and associate corresponding discrete set with the inputs
for i = 1:(NInputs-1)
	Error = VDOC_ChangeInputType(  vDBIDInp(i), SetIDs(1) );
	if Error < 0 
    	disp(' ');
        disp(' Stopped after VDOC_ChangeInputType (discrete) #');
        i
    	disp(' Error =');
    	Error
		Error = VDOC_CloseDatabase;
    	return;
	end    
end 

% Change the type of the last input to be discrete	
%	and associate corresponding discrete set with this input
Error = VDOC_ChangeInputType(  vDBIDInp(NInputs), SetIDs(2) );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_ChangeInputType (discrete) #');
    NInputs
    disp(' Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end    

% Put just the bounds for all the responses into the database
[Error,vDBIDResp] = VDOC_PutRespAll( vDBIDResp, vLBResp, vUBResp );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutRespAll.  Error =');
    Error
 	Error = VDOC_CloseDatabase;
   return;
end

% Change the type of the 2nd response (TEST) to be pass/fail
TypeRespPassFail = 3;
Error = VDOC_ChangeRespType(  vDBIDResp(2), TypeRespPassFail );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_ChangeRespType  (Pass/Fail).  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Put ONLY GOAL objective characteristic into the 1st response object (V)
[Error,vDBIDResp(1)] = VDOC_PutRespObj( vDBIDResp(1) );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutRespObj.  Error =');
    Error
 	Error = VDOC_CloseDatabase;
    return;
end

% Set the task type information 
%	1 - RSA
TaskType = 1 
Error = VDOC_PutTaskType( TaskType , 'Rounded box with discrete variables' );
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_PutTaskType.  Error =');
    Error
	Error = VDOC_CloseDatabase;
    return;
end

% Make optimization task
TaskNumber = VDOC_MakeTask;
if TaskNumber < 0 
    disp(' ');
    disp(' Stopped after VDOC_MakeTask.  Error =');
    TaskNumber 
 	Error = VDOC_CloseDatabase;
    return;
end

% Set up the optimization loop with RSA
%	StatusCode is set to initialize optimization
StatusCode = 0;
matDVs = 0;
matResps = 0;
icount = 0;

% Loop around call to optimizer and evaluation of responses
%	The loop continues for the cases of:
%	StatusCode = 0	- Initialization 
%	StatusCode = 2	- evaluate response at one point
%	StatusCode = 3	- evaluate response at several points (for gradients)
while( (StatusCode==0) | (StatusCode==2) | (StatusCode==3) )
 
    % Call Optimizer
    [Error, StatusCode, matDVs, matResps] = VDOC_RunRSA( TaskNumber, StatusCode, matDVs, matResps );
 	if Error < 0 
    	disp(' ');
    	disp(' Stopped after VDOC_RunRSA.  Error =');
    	Error
    	return;
	end
   
    % If optimization is completed (StatusCode=1), or if some unexpected 
    %	error occured - exit.
    if ( (StatusCode==1) & (StatusCode==6) )
        StatusCode
        break;
    end
    
    
    % Check if we need to evaluate responses:
	%				1st response: V
	%				2nd response: TEST 
	%				3rd response: S  
    if ( StatusCode == 2 )
    
    	% Get the number of points where we need to evaluate responses:
    	%	number of rows (first dimension) in the matrix of design variables
    	NSize = size( matDVs );
    	NPoints = NSize(1);
    
    	% For each point evaluate responses and write them into the matrix of responses
   		for i = 1:NPoints
        	% Extract design variables
        	L0 = matDVs(i,1);
        	L1 = matDVs(i,2);
        	L2 = matDVs(i,3);
        	R  = matDVs(i,4);
        
            % Evaluate responses
            V = L1*( L0*L2 - 4*R*R + pi*R*R );
			S = 2*( L0*L2 + (pi-4)*R*R ) + L1*( 2*L0 + 2*L2 + 2*(pi-4)*R );

  			if( ((L0-2*R)>=0) & ((L2-2*R)>=0) )
				TEST = -1;
			else
                V    = 0;
                TEST = 1;
				S    = 0;   
			end          
        
        	% Put responses into the corresponding rows of the matrix of responses
        	matResps(i,1) = V;
        	matResps(i,2) = TEST;
        	matResps(i,3) = S;
        end 
        
        disp(' ');
        disp(' Counting ');
        icount
        disp(' Status Code');
        StatusCode
		disp(' ====================================');
    	matDVs
   		disp(' -----');
     	matResps    
	end
end  

disp(' ');
disp(' ====================================');
disp(' Optimization is completed!!!');
StatusCode
disp(' ====================================');



% Get the results of the optimization
[Error, BestTask, BestObj, WorstConstr, BestIterNumber, BestSubIterNumber, StopCode, RunStatus] = VDOC_GetFinalResultsOptim( TaskNumber )
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_GetFinalResultsOptim.  Error =');
    Error
    return;
end


% Get the point number where the best design occured
[Error, PointNumber, Obj, Constr, BestPointNumber, BestObj, WorstConstr] = VDOC_GetSubIter( BestTask, BestIterNumber, BestSubIterNumber )
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_GetSubIter.  Error =');
    Error
    return;
end

% Get the values of the design variables and responses at the best point
[Error, vDVs, vResps] = VDOC_GetDPoint( BestTask, PointNumber )
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_GetDPoint.  Error =');
    Error
    return;
end

% Close the database
Error = VDOC_CloseDatabase;
if Error < 0 
    disp(' ');
    disp(' Stopped after VDOC_CloseDatabase.  Error =');
    Error
    return;
end



