% ADVISOR Data file:  FC_CI67_emis.m
%
% Data source:  Scott Sluder of Oak Ridge National Laboratory
%
% Data confidence level: Excellent
%
% Notes from original Excel spreadsheet from ORNL:  
% This workbook contains an engine map for a Volkwagen 1.9L TDI engine.
% The data was gathered by Oak Ridge National Laboratory.  The engine retained the 
% alternator, and so was self-sufficient.  The battery was kept charged externally,
% so that the only electrical loads during testing were those necessary to power the
% engine controls and sensors.  (The fuel injection is electronically controlled,
% but mechanically actuated, so no high electrical loads are required for fuel injection.)
%
% The tests were conducted with a fully-warmed engine.  Measurements were made
% at the following steady-state points:
% idle, with and without commanded EGR  (apparently VW turns off EGR during extended idle periods)
% 1000 RPM no absorbed load
% 1200, 1600, 1900, 2400, 2800, 3200, 3600, 4000, and 4000 RPM at 10%, 25%, 40%, 50%, 60%, 75%, 90%, and 100% load
% additionally, no load data points were collected at 1200, 1600, and 1900 rpm.
%
% The data collected at these points was used to produce the "regular" matrix
% of data in this workbook.  The regular matrix is a requirement for using this
% data with the ADVISOR model.  Data at higher-than-attainable torque values at
% each speed are the same as the value for the highest-attainable torque value
% at each speed.  Otherwise, the data is linearly interpolated as necessary to fill
% the matrix.  The matrix is basically in increments of 200 rpm and 5 foot pounds
% of torque, with a few exceptions.  The 1900 rpm sweep has been included as it
% is the speed at which rated torque is produced.  900 rpm has also been included,
% as it is the idle speed.  The engine produces its rated power at 4000 rpm,
% while 4400 rpm is the maximum engine speed.  All units for emissions, exhaust
% flow, and fuel consumption are grams per brake kilowatt-hour.  
% Units for exhaust temperature are degrees Celcius.
%
% Exhaust gas temperature was measured at the inlet to the downpipe (downstream of manifold, EGR split,
% and turbo.
%
% Many additional measurements were made at each data point, 
% including particulate size distribution, and non-regulated compounds via FTIR.
%
% Created on:  01/05/99 
% By:  Keith Wipke, National Renewable Energy Laboratory, keith_wipke@nrel.gov
%
% Revision history at end of file.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FILE ID INFO
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fc_description='Volkswagen 1.9L Turbo Diesel Engine f/ ORNL'; % one line descriptor identifying the engine
fc_version=2003; % version of ADVISOR for which the file was generated
fc_proprietary=0; % 0=> non-proprietary, 1=> proprietary, do not distribute
fc_validation=2; % 0=> no validation, 1=> data agrees with source data, 
% 2=> data matches source data and data collection methods have been verified
fc_fuel_type='Diesel';
fc_disp=1.9; % (L) engine displacement
fc_emis=1;      % boolean 0=no emis data; 1=emis data
fc_cold=1;      % boolean 0=no cold data; 1=cold data exists
disp(['Data loaded: FC_CI67_emis.m - ',fc_description]);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SPEED & TORQUE RANGES over which data is defined
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (rad/s), speed range of the engine (converted from RPM)
fc_map_spd=[800:100:4400]*pi/30;

% (N*m), torque range of the engine (converted from ft-lbf)
fc_map_trq=[165:-5:0]*(1/(3.281*0.224809)); 
fc_map_trq=fliplr(fc_map_trq);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FUEL USE AND EMISSIONS MAPS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (g/s) , fuel use map indexed vertically by fc_map_spd and 
% horizontally by fc_map_trq
fc_fuel_map=[1.59	1.63	1.71	1.81	1.89	1.95	1.99	2.04	2.13	2.25	2.4	2.55	2.71	2.87	3.02	3.15	3.25	3.34	3.44	3.54	3.65	3.76	3.86	3.96	4.05	4.12	4.18	4.24	4.31	4.4	4.49	4.58	4.67	4.75	4.83	4.9	4.93
1.53	1.58	1.67	1.78	1.88	1.94	1.99	2.04	2.12	2.24	2.38	2.54	2.69	2.85	3	3.14	3.25	3.34	3.44	3.54	3.65	3.76	3.86	3.96	4.05	4.13	4.2	4.27	4.35	4.44	4.52	4.61	4.69	4.77	4.84	4.9	4.93
1.44	1.5	1.61	1.73	1.84	1.92	1.97	2.02	2.1	2.21	2.35	2.5	2.66	2.82	2.98	3.11	3.23	3.33	3.43	3.54	3.65	3.75	3.86	3.96	4.06	4.15	4.24	4.32	4.41	4.49	4.57	4.65	4.72	4.79	4.86	4.91	4.94
1.37	1.43	1.54	1.67	1.79	1.87	1.93	1.99	2.07	2.18	2.31	2.46	2.61	2.77	2.93	3.07	3.19	3.3	3.41	3.52	3.64	3.74	3.85	3.96	4.06	4.17	4.27	4.37	4.47	4.55	4.63	4.7	4.76	4.83	4.88	4.92	4.94
1.32	1.37	1.47	1.59	1.71	1.8	1.87	1.94	2.02	2.13	2.26	2.4	2.55	2.71	2.87	3.01	3.14	3.25	3.37	3.49	3.6	3.72	3.83	3.94	4.06	4.17	4.29	4.4	4.51	4.6	4.68	4.75	4.81	4.87	4.91	4.94	4.95
1.29	1.33	1.41	1.51	1.61	1.7	1.79	1.87	1.96	2.07	2.19	2.33	2.47	2.63	2.78	2.93	3.06	3.18	3.3	3.43	3.55	3.66	3.78	3.9	4.03	4.15	4.28	4.41	4.53	4.62	4.71	4.78	4.85	4.9	4.93	4.95	4.96
1.27	1.29	1.35	1.42	1.51	1.6	1.69	1.79	1.89	2	2.12	2.25	2.39	2.53	2.69	2.83	2.96	3.09	3.22	3.34	3.47	3.59	3.71	3.84	3.98	4.12	4.26	4.39	4.51	4.62	4.71	4.8	4.87	4.92	4.94	4.96	4.96
1.25	1.27	1.3	1.35	1.42	1.51	1.61	1.71	1.82	1.93	2.05	2.17	2.3	2.44	2.58	2.72	2.85	2.98	3.11	3.24	3.36	3.49	3.63	3.76	3.91	4.05	4.2	4.34	4.47	4.59	4.69	4.79	4.86	4.92	4.94	4.95	4.95
1.24	1.25	1.27	1.3	1.35	1.43	1.53	1.64	1.75	1.86	1.98	2.1	2.22	2.35	2.49	2.62	2.75	2.87	3	3.12	3.25	3.38	3.52	3.66	3.81	3.96	4.11	4.26	4.4	4.52	4.64	4.74	4.83	4.89	4.91	4.92	4.93
1.24	1.24	1.25	1.27	1.31	1.37	1.46	1.57	1.68	1.8	1.91	2.03	2.15	2.28	2.41	2.54	2.66	2.77	2.89	3.01	3.14	3.26	3.4	3.54	3.68	3.83	3.99	4.14	4.28	4.42	4.55	4.67	4.76	4.82	4.85	4.86	4.86
1.23	1.23	1.23	1.24	1.27	1.33	1.41	1.51	1.62	1.74	1.85	1.96	2.08	2.21	2.34	2.46	2.58	2.69	2.8	2.91	3.03	3.15	3.27	3.39	3.53	3.67	3.83	3.99	4.14	4.29	4.44	4.56	4.66	4.72	4.74	4.75	4.74
1.21	1.21	1.21	1.22	1.24	1.29	1.36	1.46	1.57	1.68	1.79	1.9	2.01	2.13	2.26	2.38	2.5	2.61	2.72	2.83	2.93	3.03	3.14	3.24	3.36	3.5	3.66	3.82	3.99	4.15	4.3	4.43	4.53	4.58	4.6	4.59	4.59
1.17	1.17	1.17	1.18	1.2	1.25	1.32	1.41	1.51	1.62	1.72	1.82	1.93	2.05	2.18	2.3	2.41	2.52	2.63	2.74	2.84	2.93	3.01	3.11	3.21	3.34	3.5	3.66	3.83	3.99	4.15	4.28	4.37	4.42	4.43	4.42	4.41
1.13	1.13	1.13	1.14	1.16	1.2	1.27	1.35	1.45	1.55	1.65	1.75	1.85	1.97	2.08	2.2	2.31	2.42	2.53	2.64	2.73	2.82	2.9	2.98	3.09	3.22	3.36	3.52	3.68	3.84	3.99	4.11	4.2	4.24	4.25	4.24	4.23
1.07	1.07	1.07	1.08	1.1	1.15	1.21	1.29	1.38	1.48	1.58	1.67	1.77	1.88	1.99	2.1	2.2	2.3	2.41	2.52	2.62	2.7	2.78	2.87	2.98	3.1	3.25	3.4	3.55	3.69	3.83	3.95	4.03	4.08	4.09	4.09	4.08
1.01	1.01	1.01	1.02	1.05	1.09	1.15	1.23	1.32	1.41	1.51	1.6	1.7	1.8	1.91	2	2.09	2.18	2.28	2.39	2.49	2.59	2.67	2.77	2.87	2.99	3.13	3.27	3.41	3.54	3.67	3.78	3.87	3.92	3.95	3.97	3.97
0.95	0.95	0.96	0.97	0.99	1.03	1.09	1.16	1.25	1.34	1.43	1.53	1.62	1.72	1.82	1.9	1.99	2.07	2.16	2.26	2.37	2.47	2.56	2.66	2.76	2.87	3	3.13	3.26	3.39	3.51	3.63	3.72	3.78	3.83	3.87	3.89
0.9	0.9	0.91	0.91	0.94	0.97	1.03	1.1	1.18	1.27	1.36	1.45	1.54	1.63	1.72	1.8	1.88	1.95	2.04	2.14	2.25	2.35	2.45	2.54	2.64	2.75	2.86	2.98	3.1	3.23	3.35	3.47	3.56	3.64	3.71	3.77	3.8
0.85	0.85	0.86	0.87	0.89	0.92	0.98	1.04	1.12	1.2	1.29	1.37	1.45	1.54	1.62	1.7	1.76	1.83	1.92	2.01	2.12	2.23	2.33	2.43	2.52	2.62	2.73	2.84	2.95	3.07	3.19	3.31	3.41	3.5	3.59	3.66	3.7
0.81	0.81	0.81	0.82	0.84	0.87	0.93	0.99	1.06	1.14	1.22	1.29	1.37	1.44	1.52	1.59	1.65	1.71	1.79	1.89	2	2.11	2.21	2.31	2.41	2.51	2.6	2.71	2.81	2.92	3.03	3.14	3.24	3.34	3.44	3.52	3.57
0.76	0.76	0.77	0.77	0.79	0.83	0.88	0.94	1.01	1.08	1.15	1.21	1.28	1.35	1.42	1.48	1.54	1.6	1.68	1.77	1.88	1.99	2.1	2.2	2.3	2.39	2.49	2.58	2.68	2.78	2.88	2.97	3.07	3.17	3.27	3.35	3.4
0.72	0.72	0.72	0.73	0.75	0.78	0.83	0.89	0.95	1.01	1.07	1.13	1.19	1.26	1.33	1.39	1.44	1.5	1.57	1.66	1.76	1.88	1.99	2.1	2.19	2.29	2.38	2.46	2.55	2.64	2.73	2.82	2.91	3	3.1	3.17	3.22
0.68	0.68	0.68	0.69	0.7	0.73	0.78	0.83	0.89	0.94	1	1.05	1.11	1.17	1.23	1.29	1.34	1.4	1.47	1.55	1.66	1.77	1.89	1.99	2.09	2.18	2.26	2.35	2.43	2.51	2.59	2.67	2.75	2.84	2.93	3	3.04
0.63	0.63	0.63	0.64	0.66	0.68	0.72	0.77	0.82	0.87	0.91	0.96	1.01	1.07	1.14	1.2	1.25	1.3	1.37	1.45	1.55	1.67	1.78	1.89	1.98	2.06	2.14	2.22	2.3	2.38	2.46	2.53	2.61	2.69	2.77	2.83	2.87
0.58	0.58	0.58	0.59	0.6	0.63	0.66	0.7	0.74	0.79	0.83	0.88	0.93	0.99	1.05	1.1	1.16	1.21	1.28	1.36	1.45	1.56	1.66	1.77	1.86	1.94	2.01	2.09	2.17	2.24	2.32	2.4	2.47	2.55	2.62	2.68	2.71
0.51	0.51	0.52	0.53	0.54	0.57	0.6	0.63	0.67	0.72	0.76	0.8	0.85	0.91	0.96	1.02	1.07	1.13	1.19	1.27	1.35	1.45	1.55	1.64	1.73	1.8	1.88	1.95	2.03	2.11	2.19	2.26	2.34	2.41	2.47	2.53	2.56
0.43	0.44	0.45	0.47	0.48	0.51	0.53	0.57	0.6	0.64	0.69	0.73	0.78	0.83	0.88	0.93	0.99	1.05	1.12	1.19	1.26	1.34	1.43	1.51	1.6	1.67	1.74	1.82	1.9	1.98	2.06	2.13	2.2	2.27	2.33	2.38	2.41
0.35	0.36	0.38	0.4	0.42	0.45	0.47	0.5	0.53	0.57	0.61	0.65	0.7	0.75	0.8	0.85	0.9	0.96	1.03	1.1	1.17	1.24	1.32	1.39	1.47	1.54	1.61	1.69	1.76	1.84	1.92	2	2.06	2.13	2.19	2.24	2.27
0.28	0.29	0.32	0.34	0.37	0.39	0.41	0.44	0.47	0.5	0.54	0.58	0.62	0.67	0.72	0.76	0.82	0.88	0.94	1.01	1.07	1.13	1.2	1.27	1.34	1.41	1.48	1.56	1.63	1.71	1.79	1.86	1.93	1.99	2.05	2.09	2.12
0.22	0.23	0.26	0.29	0.32	0.34	0.36	0.38	0.41	0.44	0.47	0.51	0.56	0.6	0.65	0.69	0.74	0.79	0.85	0.91	0.97	1.03	1.09	1.16	1.23	1.3	1.36	1.43	1.51	1.58	1.66	1.73	1.8	1.86	1.91	1.95	1.97
0.18	0.2	0.22	0.26	0.28	0.31	0.32	0.35	0.37	0.4	0.43	0.47	0.51	0.56	0.6	0.64	0.68	0.72	0.78	0.83	0.88	0.94	1	1.07	1.14	1.2	1.27	1.34	1.41	1.48	1.55	1.62	1.68	1.73	1.77	1.8	1.82
0.16	0.17	0.2	0.23	0.25	0.28	0.3	0.32	0.34	0.37	0.4	0.44	0.49	0.53	0.58	0.61	0.65	0.69	0.73	0.78	0.83	0.88	0.94	1.01	1.07	1.14	1.2	1.27	1.34	1.41	1.47	1.53	1.58	1.62	1.66	1.68	1.69
0.15	0.16	0.18	0.21	0.23	0.26	0.28	0.31	0.33	0.36	0.39	0.43	0.48	0.53	0.57	0.6	0.63	0.67	0.71	0.75	0.8	0.85	0.91	0.98	1.04	1.11	1.17	1.24	1.31	1.37	1.43	1.48	1.52	1.55	1.58	1.59	1.6
0.15	0.16	0.17	0.2	0.22	0.25	0.27	0.3	0.32	0.35	0.39	0.43	0.47	0.52	0.57	0.6	0.63	0.66	0.7	0.74	0.79	0.84	0.91	0.97	1.03	1.1	1.16	1.23	1.3	1.36	1.41	1.46	1.49	1.52	1.54	1.55	1.56]';
fc_fuel_map=fliplr(fc_fuel_map);

% (g/s), engine out HC emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_hc_map=1e-3*[8.95	9.77	11.21	12.88	14.35	15.28	15.52	14.93	13.46	11.24	8.81	6.93	6.16	6.5	7.5	8.56	9.33	9.78	10.02	10.14	10.11	9.9	9.5	9.02	8.6	8.34	8.26	8.32	8.46	8.73	9.26	10.38	12.36	15.17	18.29	20.96	22.47
7.93	8.85	10.48	12.36	14.03	15.09	15.41	14.85	13.4	11.19	8.78	6.91	6.15	6.51	7.52	8.59	9.37	9.82	10.05	10.15	10.12	9.9	9.5	9.02	8.61	8.36	8.31	8.4	8.6	8.97	9.62	10.86	12.87	15.62	18.61	21.14	22.57
6.44	7.49	9.34	11.49	13.41	14.66	15.09	14.61	13.21	11.05	8.68	6.86	6.14	6.55	7.59	8.69	9.48	9.92	10.13	10.2	10.14	9.89	9.48	9.01	8.61	8.4	8.38	8.53	8.86	9.44	10.38	11.86	13.97	16.6	19.31	21.54	22.78
5.13	6.21	8.13	10.39	12.44	13.85	14.44	14.1	12.83	10.77	8.51	6.78	6.15	6.63	7.74	8.89	9.7	10.12	10.28	10.29	10.17	9.88	9.46	8.99	8.61	8.43	8.45	8.69	9.2	10.1	11.48	13.35	15.61	18.05	20.34	22.12	23.09
4.25	5.23	6.98	9.08	11.08	12.58	13.37	13.27	12.22	10.36	8.26	6.68	6.18	6.78	7.99	9.2	10.02	10.41	10.49	10.41	10.21	9.88	9.43	8.97	8.61	8.46	8.52	8.84	9.55	10.82	12.7	15.02	17.46	19.7	21.52	22.79	23.44
3.73	4.48	5.86	7.6	9.39	10.92	11.95	12.18	11.45	9.85	7.97	6.57	6.23	6.96	8.28	9.56	10.39	10.73	10.72	10.55	10.27	9.88	9.42	8.97	8.64	8.5	8.6	8.98	9.87	11.45	13.78	16.49	19.1	21.16	22.56	23.38	23.75
3.38	3.87	4.81	6.09	7.6	9.14	10.41	11	10.62	9.32	7.68	6.48	6.29	7.15	8.57	9.91	10.75	11.04	10.95	10.69	10.34	9.91	9.44	9	8.69	8.59	8.71	9.14	10.12	11.91	14.5	17.46	20.15	22.08	23.19	23.71	23.9
3.13	3.4	3.95	4.82	6.05	7.54	8.97	9.87	9.81	8.81	7.41	6.39	6.34	7.31	8.8	10.18	11.03	11.3	11.16	10.84	10.44	9.98	9.5	9.06	8.79	8.71	8.87	9.32	10.34	12.17	14.83	17.83	20.49	22.31	23.26	23.61	23.71
2.98	3.09	3.38	3.94	4.91	6.29	7.77	8.85	9.05	8.31	7.13	6.29	6.35	7.39	8.93	10.35	11.22	11.48	11.34	11	10.58	10.1	9.6	9.17	8.91	8.88	9.06	9.53	10.51	12.25	14.75	17.55	20.02	21.68	22.51	22.8	22.86
2.89	2.93	3.07	3.44	4.2	5.41	6.84	7.99	8.36	7.83	6.84	6.14	6.29	7.37	8.94	10.37	11.26	11.56	11.44	11.12	10.71	10.22	9.72	9.29	9.05	9.04	9.26	9.72	10.6	12.1	14.24	16.62	18.72	20.11	20.8	21.03	21.07
2.84	2.85	2.92	3.17	3.78	4.82	6.15	7.29	7.76	7.38	6.53	5.92	6.12	7.2	8.74	10.17	11.07	11.42	11.36	11.11	10.74	10.29	9.8	9.39	9.16	9.19	9.42	9.86	10.59	11.74	13.35	15.12	16.68	17.7	18.19	18.33	18.34
2.78	2.78	2.83	3.01	3.51	4.43	5.65	6.75	7.25	6.95	6.19	5.63	5.81	6.83	8.29	9.65	10.56	10.96	11	10.85	10.57	10.19	9.77	9.41	9.24	9.29	9.54	9.93	10.49	11.27	12.3	13.42	14.38	14.98	15.22	15.24	15.2
2.71	2.71	2.75	2.9	3.34	4.16	5.27	6.28	6.76	6.49	5.79	5.26	5.4	6.3	7.61	8.85	9.71	10.16	10.3	10.28	10.13	9.88	9.59	9.36	9.28	9.39	9.65	9.99	10.39	10.85	11.39	11.95	12.39	12.6	12.59	12.46	12.35
2.64	2.64	2.67	2.81	3.2	3.93	4.91	5.8	6.21	5.95	5.32	4.83	4.93	5.69	6.83	7.92	8.71	9.17	9.39	9.48	9.47	9.4	9.31	9.27	9.33	9.51	9.78	10.08	10.35	10.6	10.83	11.02	11.11	11.04	10.82	10.55	10.36
2.59	2.59	2.62	2.75	3.08	3.69	4.5	5.22	5.53	5.3	4.76	4.36	4.47	5.13	6.11	7.07	7.77	8.2	8.46	8.63	8.75	8.88	9.02	9.19	9.41	9.67	9.94	10.18	10.37	10.49	10.55	10.55	10.45	10.22	9.87	9.5	9.25
2.55	2.56	2.58	2.68	2.94	3.41	4.01	4.53	4.73	4.54	4.14	3.88	4.05	4.69	5.58	6.44	7.07	7.47	7.72	7.93	8.17	8.46	8.8	9.16	9.5	9.81	10.06	10.26	10.38	10.42	10.38	10.28	10.09	9.8	9.41	9.01	8.76
2.5	2.51	2.53	2.59	2.77	3.06	3.44	3.75	3.86	3.72	3.48	3.4	3.68	4.36	5.23	6.05	6.65	7.04	7.32	7.58	7.9	8.3	8.74	9.18	9.56	9.87	10.11	10.26	10.31	10.26	10.12	9.94	9.7	9.41	9.06	8.73	8.51
2.42	2.42	2.43	2.46	2.55	2.69	2.86	3	3.04	2.96	2.88	2.96	3.36	4.1	4.99	5.82	6.45	6.92	7.31	7.71	8.13	8.54	8.93	9.28	9.58	9.84	10.04	10.15	10.15	10	9.77	9.49	9.23	8.97	8.72	8.49	8.35
2.3	2.3	2.3	2.31	2.32	2.34	2.37	2.38	2.37	2.35	2.38	2.59	3.09	3.87	4.78	5.65	6.38	7.04	7.73	8.41	8.97	9.33	9.49	9.55	9.62	9.74	9.89	9.97	9.92	9.71	9.38	9.04	8.76	8.56	8.42	8.31	8.25
2.17	2.17	2.17	2.16	2.13	2.08	2.02	1.97	1.93	1.93	2.03	2.31	2.86	3.65	4.57	5.48	6.37	7.35	8.47	9.58	10.37	10.62	10.41	10.03	9.74	9.65	9.71	9.77	9.69	9.44	9.07	8.71	8.43	8.28	8.23	8.22	8.22
2.06	2.06	2.05	2.03	1.99	1.92	1.83	1.74	1.69	1.7	1.82	2.12	2.66	3.45	4.36	5.32	6.37	7.7	9.32	10.91	11.94	12.08	11.48	10.6	9.92	9.6	9.56	9.58	9.5	9.24	8.87	8.5	8.25	8.15	8.16	8.21	8.25
1.96	1.96	1.95	1.94	1.9	1.82	1.72	1.63	1.57	1.57	1.68	1.96	2.49	3.25	4.15	5.13	6.32	7.94	9.97	11.94	13.18	13.23	12.32	11.06	10.06	9.56	9.44	9.44	9.35	9.1	8.74	8.39	8.17	8.1	8.15	8.23	8.3
1.87	1.87	1.87	1.85	1.81	1.74	1.64	1.55	1.48	1.47	1.56	1.82	2.32	3.06	3.93	4.92	6.19	7.96	10.22	12.39	13.73	13.75	12.68	11.24	10.09	9.51	9.35	9.34	9.25	9	8.66	8.33	8.12	8.08	8.15	8.26	8.33
1.77	1.77	1.76	1.75	1.71	1.64	1.55	1.45	1.38	1.37	1.44	1.68	2.16	2.86	3.71	4.7	6	7.82	10.11	12.31	13.65	13.67	12.61	11.17	10.03	9.43	9.27	9.25	9.17	8.94	8.61	8.29	8.1	8.06	8.14	8.26	8.34
1.64	1.64	1.63	1.61	1.57	1.51	1.42	1.34	1.27	1.26	1.32	1.55	2	2.67	3.51	4.51	5.83	7.65	9.89	12.02	13.31	13.35	12.38	11.03	9.94	9.36	9.19	9.18	9.11	8.9	8.59	8.29	8.1	8.07	8.14	8.25	8.33
1.49	1.48	1.47	1.45	1.41	1.35	1.28	1.21	1.16	1.15	1.21	1.43	1.86	2.5	3.33	4.35	5.72	7.57	9.79	11.83	13.06	13.12	12.23	10.96	9.89	9.3	9.11	9.11	9.06	8.88	8.6	8.32	8.15	8.1	8.16	8.24	8.3
1.36	1.35	1.32	1.29	1.24	1.19	1.14	1.08	1.05	1.04	1.11	1.32	1.72	2.34	3.15	4.21	5.65	7.58	9.83	11.87	13.09	13.17	12.29	11.01	9.89	9.23	9.01	9.01	8.99	8.85	8.61	8.37	8.2	8.14	8.17	8.23	8.27
1.29	1.26	1.22	1.17	1.12	1.06	1.02	0.97	0.94	0.94	1	1.2	1.58	2.18	2.98	4.04	5.53	7.55	9.9	12.04	13.33	13.44	12.52	11.14	9.88	9.13	8.85	8.85	8.87	8.78	8.59	8.39	8.24	8.17	8.17	8.2	8.22
1.28	1.25	1.18	1.11	1.04	0.98	0.93	0.89	0.86	0.85	0.9	1.08	1.45	2.03	2.79	3.82	5.3	7.38	9.88	12.21	13.67	13.85	12.85	11.3	9.87	8.98	8.64	8.64	8.69	8.66	8.52	8.36	8.23	8.16	8.14	8.14	8.14
1.32	1.27	1.19	1.1	1.02	0.95	0.89	0.83	0.79	0.77	0.81	0.98	1.34	1.89	2.61	3.58	5.02	7.12	9.75	12.31	13.99	14.26	13.2	11.47	9.84	8.81	8.42	8.41	8.49	8.5	8.42	8.3	8.19	8.12	8.08	8.06	8.05
1.36	1.31	1.22	1.12	1.03	0.95	0.87	0.8	0.75	0.71	0.75	0.92	1.27	1.8	2.48	3.38	4.75	6.84	9.58	12.35	14.24	14.6	13.49	11.6	9.81	8.66	8.22	8.2	8.31	8.35	8.31	8.23	8.14	8.07	8.02	7.99	7.98
1.39	1.34	1.25	1.14	1.05	0.96	0.87	0.79	0.72	0.68	0.71	0.88	1.23	1.75	2.41	3.26	4.58	6.66	9.46	12.36	14.39	14.82	13.68	11.69	9.78	8.56	8.08	8.07	8.18	8.25	8.23	8.17	8.09	8.03	7.98	7.94	7.92
1.4	1.35	1.26	1.16	1.07	0.97	0.88	0.79	0.71	0.66	0.7	0.86	1.21	1.73	2.38	3.21	4.5	6.56	9.4	12.36	14.47	14.93	13.77	11.73	9.77	8.51	8.02	8	8.12	8.19	8.18	8.13	8.06	8	7.95	7.92	7.89
1.41	1.36	1.27	1.17	1.08	0.98	0.88	0.79	0.7	0.66	0.69	0.86	1.21	1.73	2.37	3.19	4.48	6.53	9.38	12.36	14.49	14.97	13.8	11.75	9.77	8.5	7.99	7.97	8.09	8.17	8.16	8.11	8.05	7.99	7.94	7.91	7.88]'; 
fc_hc_map=fliplr(fc_hc_map);

% (g/s), engine out CO emissions indexed vertically by fc_map_spd and
fc_co_map=1e-3*[27.01	24.64	20.48	15.66	11.4	8.59	7.27	7.1	7.6	8.51	9.61	10.74	11.79	12.77	13.66	14.45	15.17	15.92	16.78	17.69	18.52	19.14	19.56	19.88	20.21	20.66	21.3	22.23	23.45	24.93	26.55	28.22	29.93	31.72	33.42	34.81	35.58
29.94	27.25	22.52	17.02	12.17	8.93	7.38	7.1	7.56	8.44	9.5	10.61	11.66	12.66	13.57	14.39	15.14	15.91	16.78	17.7	18.52	19.13	19.56	19.9	20.3	20.87	21.69	22.78	24.1	25.57	27.1	28.66	30.28	31.97	33.59	34.9	35.63
34.19	31.09	25.62	19.25	13.56	9.68	7.71	7.19	7.51	8.3	9.3	10.38	11.42	12.43	13.38	14.24	15.03	15.83	16.72	17.64	18.46	19.07	19.52	19.9	20.42	21.2	22.3	23.65	25.12	26.61	28.05	29.49	30.97	32.5	33.95	35.1	35.73
37.91	34.63	28.82	21.95	15.66	11.13	8.55	7.53	7.52	8.12	9.04	10.07	11.1	12.12	13.08	13.97	14.78	15.61	16.51	17.42	18.23	18.85	19.33	19.8	20.47	21.47	22.86	24.49	26.15	27.71	29.15	30.55	31.93	33.28	34.48	35.4	35.89
40.37	37.32	31.85	25.2	18.76	13.64	10.2	8.32	7.69	7.96	8.75	9.73	10.73	11.73	12.68	13.56	14.37	15.2	16.07	16.95	17.74	18.38	18.92	19.51	20.34	21.57	23.21	25.06	26.9	28.58	30.13	31.59	32.96	34.14	35.09	35.74	36.07
41.84	39.41	34.92	29.16	23.03	17.38	12.81	9.69	8.13	7.89	8.47	9.39	10.35	11.3	12.2	13.04	13.82	14.6	15.41	16.23	16.97	17.62	18.24	18.96	19.98	21.41	23.24	25.25	27.22	29.04	30.76	32.38	33.79	34.88	35.61	36.04	36.23
42.87	41.22	38.03	33.5	27.95	21.89	16.12	11.57	8.86	7.95	8.25	9.06	9.98	10.87	11.7	12.48	13.19	13.9	14.62	15.35	16.04	16.7	17.38	18.22	19.38	20.96	22.91	25	27.05	29	30.91	32.72	34.26	35.33	35.94	36.22	36.32
43.66	42.73	40.76	37.47	32.64	26.4	19.64	13.75	9.86	8.18	8.11	8.79	9.64	10.46	11.23	11.93	12.57	13.2	13.84	14.49	15.13	15.78	16.5	17.4	18.62	20.25	22.21	24.3	26.38	28.45	30.57	32.61	34.32	35.46	36.05	36.26	36.32
44.04	43.61	42.52	40.24	36.14	30.04	22.75	15.88	10.96	8.54	8.07	8.59	9.37	10.14	10.83	11.47	12.05	12.61	13.18	13.78	14.38	15.01	15.73	16.63	17.82	19.36	21.2	23.19	25.26	27.46	29.83	32.14	34.06	35.3	35.91	36.11	36.15
43.57	43.41	42.84	41.26	37.85	32.18	24.84	17.5	11.9	8.91	8.1	8.48	9.21	9.93	10.57	11.14	11.67	12.18	12.71	13.26	13.84	14.45	15.13	15.96	17.03	18.39	20.01	21.84	23.89	26.25	28.88	31.47	33.57	34.9	35.51	35.69	35.71
41.42	41.37	41.06	39.94	37.16	32.15	25.26	18.06	12.33	9.12	8.15	8.47	9.16	9.85	10.43	10.96	11.45	11.93	12.43	12.96	13.49	14.06	14.69	15.42	16.32	17.43	18.8	20.44	22.47	24.98	27.85	30.66	32.89	34.22	34.79	34.9	34.88
36.92	36.9	36.72	35.91	33.71	29.54	23.58	17.18	11.97	9.04	8.17	8.52	9.21	9.87	10.42	10.91	11.36	11.83	12.31	12.8	13.31	13.82	14.37	14.99	15.71	16.59	17.69	19.15	21.13	23.74	26.78	29.71	31.97	33.26	33.71	33.71	33.62
30.01	30.01	29.89	29.33	27.7	24.54	19.92	14.91	10.85	8.64	8.13	8.6	9.3	9.94	10.46	10.92	11.35	11.79	12.26	12.74	13.21	13.67	14.15	14.65	15.22	15.9	16.8	18.08	19.97	22.57	25.64	28.6	30.82	32.01	32.33	32.2	32.02
21.97	21.97	21.9	21.54	20.47	18.35	15.25	11.92	9.32	8.08	8.04	8.66	9.39	10.01	10.5	10.93	11.33	11.76	12.21	12.68	13.13	13.56	13.97	14.38	14.84	15.39	16.13	17.26	19.02	21.52	24.5	27.38	29.51	30.6	30.82	30.59	30.36
14.57	14.57	14.53	14.33	13.74	12.56	10.85	9.09	7.86	7.52	7.93	8.7	9.44	10.03	10.49	10.89	11.26	11.65	12.1	12.56	13.01	13.42	13.79	14.16	14.55	15.03	15.67	16.69	18.32	20.66	23.47	26.18	28.17	29.18	29.36	29.12	28.89
9.28	9.28	9.27	9.18	8.92	8.42	7.71	7.08	6.83	7.14	7.85	8.69	9.42	9.99	10.41	10.77	11.11	11.47	11.89	12.35	12.81	13.22	13.6	13.96	14.33	14.78	15.38	16.32	17.83	20	22.6	25.09	26.93	27.88	28.1	27.97	27.8
6.31	6.31	6.31	6.3	6.23	6.12	5.99	6	6.3	6.94	7.79	8.65	9.34	9.87	10.27	10.59	10.89	11.22	11.62	12.07	12.54	12.99	13.41	13.81	14.2	14.64	15.2	16.07	17.47	19.48	21.85	24.11	25.8	26.75	27.09	27.13	27.09
5.08	5.08	5.09	5.1	5.13	5.19	5.33	5.62	6.13	6.88	7.74	8.56	9.21	9.71	10.09	10.39	10.67	10.97	11.33	11.77	12.27	12.78	13.28	13.75	14.18	14.61	15.12	15.91	17.18	19.01	21.16	23.22	24.8	25.79	26.3	26.54	26.64
4.73	4.73	4.74	4.77	4.84	4.97	5.21	5.59	6.16	6.9	7.69	8.44	9.05	9.53	9.91	10.22	10.49	10.77	11.1	11.53	12.04	12.62	13.21	13.77	14.25	14.67	15.13	15.82	16.96	18.6	20.54	22.42	23.93	24.98	25.66	26.08	26.3
4.76	4.76	4.77	4.8	4.89	5.06	5.33	5.73	6.28	6.95	7.66	8.33	8.9	9.39	9.8	10.14	10.42	10.68	10.98	11.38	11.91	12.54	13.22	13.86	14.38	14.79	15.19	15.8	16.8	18.27	20.03	21.77	23.22	24.31	25.11	25.66	25.96
4.95	4.95	4.96	5	5.09	5.27	5.55	5.95	6.46	7.05	7.67	8.27	8.83	9.35	9.82	10.21	10.52	10.77	11.04	11.4	11.91	12.57	13.3	13.98	14.52	14.92	15.29	15.82	16.72	18.04	19.65	21.27	22.67	23.77	24.61	25.21	25.54
5.25	5.25	5.26	5.3	5.39	5.58	5.87	6.25	6.72	7.24	7.77	8.32	8.88	9.47	10.05	10.53	10.88	11.13	11.35	11.66	12.12	12.76	13.48	14.16	14.7	15.08	15.41	15.89	16.71	17.93	19.43	20.94	22.27	23.33	24.15	24.74	25.07
5.64	5.64	5.65	5.7	5.8	5.99	6.28	6.65	7.07	7.52	7.98	8.49	9.09	9.81	10.54	11.16	11.59	11.83	11.99	12.21	12.58	13.14	13.8	14.45	14.95	15.29	15.58	16.01	16.76	17.91	19.32	20.76	22	23	23.75	24.3	24.59
6.15	6.15	6.17	6.22	6.33	6.54	6.84	7.19	7.55	7.91	8.29	8.78	9.47	10.36	11.31	12.13	12.65	12.88	12.94	13.03	13.27	13.71	14.29	14.86	15.31	15.6	15.83	16.19	16.87	17.95	19.3	20.68	21.86	22.77	23.45	23.92	24.17
6.72	6.73	6.77	6.84	6.98	7.21	7.52	7.86	8.18	8.46	8.77	9.25	10.04	11.14	12.35	13.37	13.98	14.17	14.1	14.03	14.11	14.42	14.9	15.4	15.8	16.03	16.18	16.44	17.02	18.04	19.34	20.68	21.8	22.63	23.22	23.61	23.82
7.23	7.28	7.36	7.5	7.7	7.99	8.34	8.7	9.01	9.26	9.54	10.05	10.96	12.27	13.72	14.91	15.57	15.66	15.43	15.15	15.06	15.25	15.65	16.11	16.46	16.61	16.64	16.76	17.23	18.16	19.42	20.72	21.79	22.56	23.06	23.38	23.55
7.5	7.6	7.79	8.06	8.38	8.78	9.24	9.7	10.1	10.4	10.73	11.32	12.37	13.87	15.51	16.83	17.48	17.45	17.02	16.55	16.31	16.39	16.72	17.14	17.42	17.46	17.31	17.24	17.53	18.34	19.52	20.79	21.82	22.54	22.98	23.24	23.38
7.4	7.6	7.96	8.43	8.95	9.53	10.17	10.83	11.42	11.89	12.34	13.06	14.27	15.95	17.75	19.15	19.78	19.65	19.06	18.44	18.07	18.06	18.33	18.65	18.82	18.67	18.28	17.92	17.96	18.59	19.68	20.89	21.9	22.58	22.99	23.22	23.33
7.01	7.33	7.91	8.62	9.38	10.18	11.05	11.95	12.78	13.45	14.09	14.97	16.34	18.18	20.11	21.6	22.26	22.13	21.51	20.84	20.41	20.33	20.49	20.69	20.67	20.27	19.54	18.83	18.54	18.94	19.9	21.07	22.06	22.73	23.13	23.35	23.45
6.53	6.97	7.74	8.69	9.68	10.69	11.76	12.87	13.9	14.77	15.58	16.61	18.13	20.1	22.15	23.74	24.52	24.49	23.97	23.37	22.96	22.82	22.87	22.91	22.68	21.99	20.91	19.81	19.19	19.36	20.21	21.32	22.31	23	23.41	23.65	23.76
6.15	6.66	7.57	8.68	9.83	10.98	12.17	13.41	14.57	15.57	16.5	17.66	19.29	21.38	23.52	25.22	26.13	26.28	25.95	25.5	25.15	24.99	24.95	24.85	24.42	23.49	22.1	20.69	19.81	19.81	20.57	21.66	22.65	23.36	23.81	24.08	24.22
5.91	6.45	7.4	8.57	9.8	11.03	12.3	13.59	14.8	15.85	16.86	18.12	19.85	22	24.21	25.99	27.02	27.34	27.18	26.87	26.6	26.43	26.33	26.13	25.58	24.48	22.9	21.31	20.28	20.21	20.93	22.02	23.01	23.75	24.24	24.57	24.74
5.8	6.31	7.24	8.39	9.63	10.91	12.22	13.54	14.75	15.82	16.87	18.2	19.99	22.2	24.44	26.27	27.38	27.8	27.75	27.53	27.31	27.15	27.02	26.78	26.16	24.98	23.31	21.65	20.58	20.5	21.23	22.32	23.32	24.07	24.61	24.98	25.18
5.75	6.24	7.12	8.25	9.49	10.78	12.11	13.44	14.65	15.73	16.81	18.17	20	22.23	24.49	26.34	27.47	27.93	27.93	27.75	27.54	27.39	27.25	26.99	26.35	25.15	23.46	21.77	20.71	20.65	21.4	22.49	23.49	24.26	24.82	25.21	25.43]'; 
fc_co_map=fliplr(fc_co_map);

% (g/s), engine out NOx emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_nox_map=1e-3*[82.72	85.85	91.37	97.76	103.4	107.14	108.94	109.3	108.91	108.25	107.97	109.01	112.14	117.35	123.53	129.32	133.94	137.7	141.24	144.87	148.52	152.04	155.45	158.62	161.4	163.63	165.48	167.32	169.44	171.7	173.34	173.16	170.25	164.7	157.97	152.05	148.67
78.82	82.37	88.62	95.88	102.3	106.57	108.65	109.12	108.73	108.03	107.71	108.72	111.86	117.13	123.39	129.27	133.98	137.8	141.36	144.99	148.62	152.12	155.51	158.71	161.59	164.02	166.16	168.23	170.35	172.26	173.31	172.53	169.28	163.74	157.26	151.64	148.45
73.15	77.22	84.39	92.75	100.2	105.27	107.84	108.55	108.25	107.53	107.15	108.14	111.31	116.68	123.1	129.16	134.03	137.97	141.59	145.21	148.8	152.26	155.64	158.88	161.91	164.65	167.23	169.62	171.65	172.88	172.82	170.95	167.11	161.65	155.72	150.76	147.99
68.16	72.42	79.98	88.88	97	102.79	106.05	107.29	107.27	106.64	106.22	107.18	110.4	115.92	122.56	128.87	133.99	138.12	141.84	145.48	149.03	152.46	155.83	159.12	162.3	165.34	168.3	170.94	172.73	173.02	171.51	168.27	163.71	158.47	153.42	149.45	147.29
64.86	68.77	75.77	84.25	92.37	98.73	102.9	105.03	105.59	105.15	104.71	105.61	108.86	114.54	121.42	128.04	133.47	137.88	141.78	145.51	149.1	152.57	156.01	159.41	162.74	166	169.21	171.92	173.32	172.59	169.6	164.98	159.76	154.84	150.8	147.96	146.5
62.89	65.96	71.6	78.77	86.29	93.04	98.32	101.69	103.06	102.92	102.42	103.16	106.34	112.07	119.13	126.01	131.78	136.58	140.8	144.78	148.57	152.27	155.97	159.65	163.22	166.7	170.03	172.67	173.61	171.97	167.73	161.98	156.23	151.62	148.48	146.65	145.81
61.54	63.59	67.52	72.99	79.52	86.42	92.71	97.33	99.58	99.77	99.2	99.72	102.69	108.27	115.28	122.24	128.26	133.42	138.05	142.45	146.68	150.92	155.24	159.52	163.58	167.39	170.85	173.41	173.99	171.69	166.6	160.12	154.03	149.63	147.1	145.92	145.5
60.54	61.67	64.03	67.84	73.23	79.89	86.71	92.2	95.18	95.72	95.15	95.48	98.13	103.34	110	116.75	122.76	128.12	133.09	137.94	142.78	147.84	153.12	158.35	163.21	167.59	171.39	174.08	174.56	172.01	166.55	159.76	153.55	149.3	147.07	146.24	146.02
59.86	60.38	61.63	64.11	68.33	74.26	80.92	86.65	90	90.82	90.33	90.52	92.86	97.64	103.81	110.16	115.93	121.21	126.26	131.39	136.77	142.66	148.98	155.29	161.13	166.34	170.83	174.07	174.98	172.75	167.57	161.03	155.07	151.03	148.96	148.23	148.06
59.44	59.63	60.23	61.79	64.93	69.83	75.74	81.07	84.29	85.03	84.4	84.34	86.39	90.86	96.75	102.83	108.39	113.52	118.5	123.68	129.31	135.69	142.69	149.82	156.54	162.69	168.19	172.48	174.46	173.3	169.21	163.66	158.47	154.86	152.91	152.11	151.85
58.97	59.02	59.32	60.28	62.52	66.32	71.14	75.54	77.99	77.99	76.64	75.96	77.62	81.95	87.87	94.06	99.74	104.99	110.06	115.29	120.95	127.38	134.55	142.01	149.33	156.38	163.09	168.81	172.41	173.01	170.8	167.01	163.13	160.14	158.17	157.01	156.45
58.04	58.05	58.2	58.77	60.28	62.99	66.51	69.52	70.59	69.18	66.52	64.8	65.87	70.02	76.08	82.58	88.68	94.4	99.91	105.4	111.11	117.46	124.58	132.22	140.08	148.05	156	163.23	168.59	171.26	171.35	169.76	167.47	165.06	162.75	160.82	159.67
56.22	56.22	56.28	56.55	57.38	58.94	60.92	62.18	61.54	58.59	54.62	51.88	52.22	55.89	61.68	68.15	74.47	80.65	86.69	92.6	98.52	104.97	112.3	120.43	129.12	138.2	147.44	156.03	162.94	167.51	169.83	170.4	169.57	167.52	164.55	161.49	159.52
53.31	53.3	53.28	53.27	53.38	53.65	53.85	53.21	50.97	46.99	42.45	39.29	39.1	42.02	46.96	52.7	58.57	64.59	70.65	76.67	82.77	89.54	97.45	106.49	116.34	126.66	137.02	146.62	154.66	160.74	164.94	167.37	167.77	165.98	162.37	158.33	155.64
49.29	49.28	49.19	48.92	48.34	47.34	45.82	43.51	40.23	36.08	31.98	29.26	28.94	31.07	34.8	39.25	43.99	49.05	54.36	59.89	65.85	72.82	81.23	90.99	101.68	112.8	123.75	133.85	142.56	149.79	155.54	159.54	161.1	159.86	156.3	152.07	149.22
44.41	44.39	44.26	43.8	42.71	40.76	37.99	34.59	30.91	27.3	24.29	22.54	22.52	24.16	26.86	30.01	33.36	37	41.02	45.56	50.96	57.72	66.07	75.77	86.27	97.07	107.58	117.38	126.23	134.23	141.24	146.65	149.51	149.46	147.07	143.91	141.73
38.92	38.9	38.76	38.23	36.93	34.59	31.31	27.61	24.11	21.31	19.45	18.7	19.14	20.61	22.69	24.95	27.21	29.6	32.38	35.89	40.54	46.68	54.28	62.94	72.11	81.41	90.47	99.16	107.51	115.73	123.48	129.93	134.09	135.75	135.49	134.44	133.61
33.18	33.17	33.04	32.55	31.32	29.09	25.98	22.6	19.65	17.63	16.63	16.6	17.42	18.94	20.79	22.62	24.26	25.87	27.82	30.56	34.53	39.91	46.44	53.62	60.94	68.19	75.25	82.33	89.67	97.49	105.31	112.23	117.4	120.77	122.83	124.11	124.77
27.6	27.59	27.49	27.1	26.1	24.27	21.73	19.01	16.76	15.41	14.97	15.35	16.38	17.95	19.72	21.38	22.77	24.05	25.61	27.94	31.41	36.09	41.63	47.47	53.15	58.59	63.9	69.45	75.59	82.55	89.84	96.63	102.25	106.77	110.49	113.46	115.19
22.64	22.63	22.55	22.26	21.51	20.14	18.24	16.25	14.67	13.86	13.8	14.4	15.53	17.08	18.79	20.35	21.63	22.79	24.2	26.31	29.45	33.62	38.43	43.35	47.95	52.2	56.33	60.77	65.92	72.01	78.58	84.95	90.57	95.5	99.91	103.55	105.69
18.58	18.57	18.52	18.31	17.77	16.77	15.41	14.02	13	12.59	12.78	13.5	14.65	16.14	17.75	19.23	20.43	21.54	22.89	24.89	27.79	31.6	35.92	40.25	44.22	47.82	51.28	55.07	59.58	65.01	71	76.93	82.31	87.15	91.48	95.02	97.08
15.41	15.41	15.37	15.22	14.82	14.1	13.12	12.16	11.51	11.37	11.72	12.48	13.6	15.02	16.52	17.9	19.03	20.1	21.4	23.32	26.07	29.64	33.64	37.6	41.17	44.35	47.4	50.79	54.92	59.97	65.62	71.25	76.39	80.96	84.95	88.12	89.92
12.88	12.87	12.84	12.73	12.43	11.9	11.18	10.51	10.1	10.12	10.53	11.29	12.34	13.65	15.03	16.29	17.35	18.36	19.61	21.45	24.09	27.49	31.29	35.01	38.29	41.15	43.89	46.98	50.84	55.67	61.11	66.55	71.48	75.8	79.48	82.34	83.94
10.61	10.6	10.58	10.49	10.27	9.88	9.36	8.9	8.67	8.81	9.26	9.98	10.94	12.09	13.29	14.41	15.36	16.3	17.49	19.23	21.73	24.99	28.64	32.19	35.27	37.89	40.36	43.2	46.87	51.53	56.81	62.09	66.84	70.96	74.45	77.13	78.63
8.39	8.39	8.37	8.3	8.15	7.88	7.55	7.29	7.25	7.5	8.01	8.7	9.54	10.5	11.5	12.43	13.28	14.15	15.25	16.85	19.14	22.16	25.61	29.02	31.96	34.43	36.72	39.39	42.89	47.4	52.5	57.59	62.14	66.08	69.4	71.97	73.41
6.23	6.22	6.19	6.13	6.05	5.92	5.78	5.74	5.88	6.27	6.83	7.49	8.22	8.99	9.77	10.54	11.3	12.14	13.19	14.63	16.65	19.34	22.5	25.71	28.54	30.91	33.1	35.64	39.01	43.32	48.21	53.05	57.38	61.12	64.29	66.76	68.15
4.36	4.33	4.27	4.21	4.18	4.17	4.2	4.35	4.64	5.1	5.68	6.3	6.92	7.55	8.17	8.83	9.56	10.42	11.45	12.76	14.5	16.8	19.61	22.55	25.24	27.52	29.63	32.06	35.26	39.35	43.99	48.58	52.68	56.2	59.19	61.52	62.84
3.03	2.96	2.85	2.77	2.74	2.8	2.94	3.18	3.53	3.98	4.49	5.03	5.56	6.09	6.63	7.24	7.96	8.86	9.92	11.15	12.67	14.64	17.1	19.74	22.23	24.38	26.37	28.65	31.64	35.48	39.85	44.21	48.1	51.42	54.19	56.35	57.57
2.31	2.2	2.04	1.9	1.85	1.91	2.07	2.3	2.59	2.94	3.34	3.77	4.21	4.69	5.2	5.78	6.49	7.39	8.44	9.62	11	12.73	14.88	17.22	19.47	21.42	23.26	25.35	28.11	31.67	35.8	39.96	43.67	46.76	49.29	51.19	52.26
2.06	1.91	1.69	1.49	1.4	1.42	1.53	1.69	1.88	2.1	2.37	2.69	3.08	3.52	4	4.55	5.22	6.06	7.06	8.18	9.47	11.03	12.94	15.02	17.02	18.78	20.45	22.35	24.87	28.17	32.04	35.95	39.43	42.27	44.49	46.11	47
2.04	1.87	1.61	1.36	1.22	1.2	1.25	1.34	1.44	1.56	1.73	1.98	2.32	2.75	3.22	3.72	4.31	5.05	5.96	7	8.21	9.66	11.4	13.27	15.06	16.65	18.18	19.94	22.3	25.37	28.95	32.54	35.68	38.16	40.02	41.32	42.02
2.08	1.89	1.61	1.34	1.17	1.12	1.14	1.18	1.22	1.29	1.41	1.63	1.95	2.36	2.81	3.28	3.8	4.45	5.27	6.24	7.4	8.77	10.4	12.13	13.79	15.27	16.72	18.43	20.69	23.6	26.89	30.1	32.8	34.85	36.33	37.33	37.85
2.1	1.91	1.62	1.34	1.16	1.1	1.1	1.13	1.15	1.2	1.3	1.51	1.82	2.23	2.67	3.12	3.6	4.18	4.94	5.87	6.99	8.33	9.89	11.56	13.16	14.59	16.01	17.71	19.96	22.78	25.86	28.72	31.01	32.65	33.77	34.51	34.89
2.11	1.92	1.62	1.34	1.16	1.09	1.1	1.12	1.14	1.18	1.28	1.48	1.8	2.2	2.64	3.08	3.54	4.11	4.84	5.75	6.86	8.18	9.73	11.37	12.95	14.36	15.78	17.5	19.76	22.54	25.5	28.17	30.21	31.6	32.51	33.1	33.4]';  
fc_nox_map=fliplr(fc_nox_map);

% (g/s), engine out PM emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_pm_map=1e-3*[2.28	2.22	2.13	2.02	1.92	1.87	1.89	2.04	2.37	2.88	3.52	4.28	5.13	6.09	7.05	7.9	8.58	9.14	9.69	10.27	10.91	11.58	12.32	13.13	14.02	15.09	16.47	18.34	20.78	23.59	26.25	28.09	28.64	27.98	26.64	25.34	24.57
2.34	2.28	2.17	2.04	1.93	1.86	1.87	2.02	2.32	2.81	3.43	4.17	5.01	5.97	6.94	7.81	8.51	9.09	9.65	10.25	10.89	11.58	12.33	13.18	14.21	15.52	17.25	19.43	21.97	24.6	26.87	28.29	28.57	27.8	26.49	25.24	24.52
2.43	2.36	2.22	2.07	1.92	1.83	1.83	1.95	2.23	2.67	3.25	3.95	4.77	5.72	6.69	7.57	8.29	8.9	9.5	10.13	10.8	11.51	12.3	13.23	14.48	16.19	18.46	21.1	23.78	26.1	27.73	28.47	28.3	27.39	26.15	25.05	24.41
2.51	2.42	2.27	2.09	1.91	1.79	1.75	1.83	2.07	2.47	3	3.65	4.42	5.32	6.26	7.11	7.83	8.47	9.12	9.8	10.52	11.27	12.12	13.17	14.66	16.8	19.61	22.7	25.49	27.43	28.36	28.38	27.74	26.73	25.64	24.75	24.26
2.56	2.47	2.31	2.11	1.91	1.74	1.65	1.68	1.87	2.22	2.7	3.29	3.98	4.79	5.63	6.42	7.12	7.77	8.44	9.17	9.93	10.74	11.67	12.85	14.58	17.09	20.33	23.76	26.58	28.18	28.52	27.95	26.97	25.94	25.05	24.41	24.08
2.59	2.51	2.37	2.17	1.95	1.74	1.57	1.53	1.65	1.95	2.37	2.89	3.49	4.18	4.9	5.59	6.23	6.86	7.53	8.25	9.02	9.86	10.85	12.17	14.11	16.91	20.46	24.07	26.87	28.24	28.21	27.31	26.17	25.2	24.52	24.11	23.92
2.62	2.56	2.44	2.27	2.04	1.78	1.55	1.42	1.46	1.68	2.04	2.48	2.99	3.57	4.16	4.74	5.3	5.88	6.51	7.18	7.92	8.74	9.75	11.14	13.23	16.2	19.88	23.54	26.28	27.54	27.41	26.49	25.42	24.6	24.11	23.87	23.78
2.64	2.6	2.53	2.39	2.17	1.88	1.58	1.36	1.31	1.46	1.74	2.12	2.54	3.02	3.52	4.01	4.5	5	5.54	6.14	6.79	7.55	8.52	9.91	12	14.96	18.56	22.09	24.73	26	26.03	25.38	24.6	24.03	23.72	23.59	23.56
2.66	2.64	2.59	2.49	2.29	1.99	1.64	1.35	1.22	1.3	1.52	1.84	2.22	2.62	3.05	3.47	3.87	4.29	4.74	5.24	5.79	6.47	7.35	8.64	10.57	13.29	16.57	19.78	22.24	23.59	23.96	23.78	23.46	23.21	23.08	23.03	23.01
2.66	2.65	2.63	2.55	2.37	2.08	1.71	1.37	1.2	1.23	1.44	1.74	2.08	2.44	2.8	3.15	3.48	3.81	4.16	4.56	5.01	5.59	6.35	7.45	9.09	11.39	14.15	16.9	19.12	20.55	21.32	21.68	21.84	21.91	21.93	21.92	21.91
2.64	2.64	2.62	2.56	2.4	2.12	1.75	1.41	1.23	1.28	1.51	1.83	2.16	2.49	2.79	3.08	3.33	3.58	3.84	4.14	4.49	4.94	5.54	6.4	7.68	9.47	11.66	13.89	15.83	17.32	18.4	19.19	19.73	20.03	20.13	20.13	20.1
2.58	2.58	2.57	2.52	2.38	2.12	1.76	1.45	1.3	1.41	1.71	2.08	2.44	2.75	3.02	3.25	3.44	3.62	3.81	4.02	4.27	4.58	4.98	5.57	6.47	7.78	9.44	11.2	12.87	14.32	15.56	16.59	17.32	17.73	17.85	17.82	17.77
2.46	2.46	2.45	2.41	2.28	2.05	1.74	1.47	1.4	1.59	1.97	2.41	2.8	3.12	3.38	3.59	3.75	3.89	4.03	4.18	4.34	4.5	4.7	5.02	5.59	6.5	7.72	9.1	10.49	11.81	13.04	14.09	14.86	15.27	15.39	15.34	15.28
2.27	2.27	2.26	2.22	2.12	1.92	1.67	1.48	1.48	1.74	2.18	2.67	3.09	3.45	3.72	3.95	4.12	4.26	4.4	4.53	4.63	4.66	4.67	4.76	5.07	5.69	6.6	7.67	8.79	9.91	10.98	11.91	12.61	12.99	13.12	13.1	13.06
2.01	2.01	2	1.98	1.89	1.74	1.55	1.43	1.5	1.8	2.25	2.75	3.2	3.58	3.9	4.16	4.36	4.54	4.71	4.86	4.94	4.89	4.77	4.69	4.83	5.28	5.99	6.85	7.74	8.61	9.44	10.18	10.74	11.1	11.28	11.36	11.39
1.72	1.71	1.71	1.69	1.63	1.52	1.39	1.34	1.44	1.74	2.17	2.63	3.07	3.47	3.82	4.11	4.36	4.58	4.81	5	5.09	5.02	4.84	4.69	4.76	5.13	5.75	6.47	7.17	7.82	8.41	8.94	9.38	9.73	10.01	10.22	10.34
1.4	1.4	1.4	1.39	1.35	1.28	1.21	1.2	1.32	1.58	1.95	2.36	2.76	3.15	3.5	3.82	4.09	4.35	4.63	4.88	5.01	4.97	4.8	4.67	4.75	5.11	5.68	6.32	6.9	7.38	7.79	8.15	8.5	8.86	9.24	9.58	9.79
1.1	1.1	1.1	1.1	1.08	1.05	1.02	1.04	1.16	1.38	1.67	2.01	2.36	2.71	3.06	3.37	3.66	3.95	4.26	4.57	4.76	4.79	4.7	4.64	4.77	5.14	5.68	6.25	6.74	7.11	7.39	7.65	7.95	8.35	8.81	9.26	9.53
0.84	0.84	0.84	0.84	0.84	0.84	0.84	0.89	0.99	1.16	1.39	1.66	1.95	2.28	2.61	2.92	3.2	3.5	3.85	4.19	4.45	4.56	4.59	4.62	4.81	5.18	5.68	6.19	6.59	6.87	7.07	7.26	7.55	7.99	8.52	9.03	9.34
0.64	0.64	0.64	0.65	0.66	0.67	0.7	0.75	0.84	0.97	1.14	1.35	1.61	1.92	2.23	2.54	2.82	3.13	3.48	3.86	4.17	4.37	4.49	4.61	4.84	5.2	5.65	6.09	6.41	6.61	6.73	6.87	7.15	7.6	8.17	8.7	9.02
0.52	0.52	0.52	0.52	0.53	0.56	0.59	0.64	0.72	0.82	0.94	1.11	1.34	1.63	1.95	2.25	2.54	2.85	3.21	3.59	3.95	4.21	4.41	4.59	4.84	5.19	5.59	5.96	6.21	6.33	6.37	6.46	6.69	7.13	7.68	8.2	8.51
0.44	0.44	0.44	0.45	0.46	0.48	0.52	0.56	0.62	0.69	0.78	0.92	1.13	1.41	1.72	2.02	2.31	2.61	2.97	3.36	3.74	4.05	4.3	4.53	4.8	5.14	5.51	5.83	6.02	6.07	6.03	6.05	6.23	6.61	7.11	7.58	7.86
0.4	0.4	0.4	0.41	0.42	0.43	0.46	0.49	0.54	0.58	0.65	0.77	0.96	1.21	1.51	1.79	2.06	2.36	2.71	3.11	3.5	3.85	4.15	4.43	4.73	5.07	5.43	5.72	5.87	5.86	5.76	5.71	5.81	6.11	6.53	6.94	7.19
0.37	0.37	0.37	0.37	0.37	0.38	0.4	0.42	0.45	0.49	0.54	0.64	0.8	1.02	1.28	1.54	1.79	2.07	2.42	2.81	3.21	3.6	3.95	4.29	4.63	5	5.36	5.65	5.78	5.73	5.58	5.45	5.48	5.69	6.04	6.38	6.59
0.33	0.33	0.33	0.34	0.34	0.34	0.34	0.35	0.37	0.4	0.44	0.52	0.66	0.84	1.06	1.28	1.52	1.79	2.12	2.5	2.9	3.31	3.71	4.1	4.51	4.92	5.31	5.61	5.75	5.68	5.47	5.28	5.23	5.37	5.65	5.93	6.11
0.29	0.3	0.3	0.3	0.3	0.3	0.3	0.3	0.31	0.33	0.37	0.43	0.54	0.69	0.86	1.05	1.27	1.54	1.86	2.22	2.61	3.01	3.44	3.89	4.35	4.8	5.25	5.59	5.75	5.66	5.42	5.16	5.04	5.12	5.35	5.6	5.76
0.25	0.26	0.27	0.27	0.28	0.28	0.27	0.26	0.27	0.28	0.32	0.37	0.46	0.57	0.71	0.87	1.08	1.33	1.64	1.97	2.32	2.71	3.14	3.62	4.11	4.62	5.12	5.53	5.73	5.66	5.38	5.08	4.91	4.95	5.14	5.37	5.52
0.2	0.21	0.22	0.24	0.25	0.25	0.25	0.24	0.24	0.26	0.28	0.33	0.39	0.48	0.58	0.72	0.91	1.15	1.43	1.72	2.02	2.37	2.79	3.26	3.78	4.34	4.91	5.4	5.67	5.64	5.37	5.04	4.85	4.86	5.03	5.24	5.38
0.14	0.15	0.18	0.2	0.22	0.23	0.23	0.23	0.23	0.24	0.26	0.3	0.34	0.41	0.49	0.6	0.75	0.95	1.18	1.42	1.68	1.98	2.36	2.81	3.34	3.94	4.6	5.2	5.57	5.61	5.37	5.06	4.86	4.85	5.01	5.21	5.35
0.09	0.1	0.13	0.16	0.18	0.2	0.21	0.21	0.22	0.23	0.24	0.27	0.31	0.35	0.41	0.49	0.6	0.75	0.92	1.1	1.31	1.57	1.91	2.34	2.87	3.51	4.26	4.97	5.45	5.57	5.39	5.11	4.92	4.92	5.07	5.26	5.39
0.05	0.06	0.08	0.11	0.14	0.16	0.18	0.19	0.2	0.22	0.23	0.26	0.28	0.32	0.36	0.41	0.48	0.58	0.69	0.82	0.99	1.21	1.52	1.93	2.46	3.14	3.96	4.76	5.34	5.53	5.41	5.17	5	5.02	5.17	5.35	5.48
0.03	0.04	0.05	0.07	0.1	0.12	0.15	0.17	0.19	0.21	0.23	0.25	0.27	0.3	0.33	0.37	0.41	0.47	0.54	0.64	0.77	0.98	1.27	1.66	2.18	2.89	3.76	4.62	5.25	5.49	5.41	5.2	5.08	5.11	5.27	5.46	5.58
0.02	0.02	0.03	0.05	0.07	0.1	0.13	0.16	0.19	0.21	0.23	0.24	0.27	0.29	0.32	0.35	0.38	0.42	0.47	0.54	0.67	0.86	1.14	1.52	2.05	2.77	3.66	4.54	5.19	5.45	5.4	5.22	5.12	5.18	5.35	5.54	5.66
0.01	0.02	0.03	0.04	0.06	0.09	0.12	0.15	0.18	0.21	0.22	0.24	0.27	0.29	0.32	0.34	0.37	0.4	0.44	0.51	0.63	0.82	1.1	1.47	2	2.72	3.62	4.51	5.16	5.43	5.38	5.22	5.14	5.21	5.4	5.59	5.71]';
fc_pm_map=fliplr(fc_pm_map);

% (g/s), engine out O2 emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_o2_map=1e-3*[2379	2541	2827	3158	3450	3651	3779	3905	4098	4382	4715	5041	5318	5547	5734	5887	6012	6125	6245	6381	6540	6723	6932	7144	7327	7456	7534	7585	7631	7683	7745	7817	7906	8010	8117	8206	8256
2178	2362	2687	3064	3397	3627	3771	3904	4101	4386	4721	5049	5329	5561	5752	5908	6035	6148	6266	6398	6553	6733	6940	7150	7333	7464	7548	7604	7654	7708	7769	7840	7925	8026	8128	8212	8259
1886	2099	2474	2911	3302	3575	3748	3898	4104	4395	4735	5069	5357	5600	5802	5969	6102	6216	6328	6452	6596	6768	6967	7171	7351	7483	7572	7636	7692	7750	7812	7883	7966	8059	8151	8225	8266
1631	1856	2255	2726	3157	3475	3691	3874	4103	4408	4757	5102	5406	5673	5901	6092	6240	6357	6461	6570	6697	6854	7038	7227	7394	7518	7606	7673	7734	7797	7866	7942	8024	8108	8185	8244	8276
1462	1671	2047	2503	2944	3303	3578	3821	4094	4425	4790	5152	5484	5789	6061	6290	6462	6585	6681	6773	6880	7016	7177	7342	7484	7588	7661	7720	7779	7846	7923	8006	8090	8164	8225	8267	8288
1361	1528	1836	2231	2651	3047	3400	3731	4072	4444	4834	5222	5591	5948	6277	6556	6759	6893	6983	7061	7150	7264	7399	7532	7641	7714	7760	7797	7842	7904	7983	8070	8151	8215	8260	8286	8298
1291	1404	1623	1933	2315	2740	3178	3612	4038	4466	4889	5309	5721	6137	6531	6866	7103	7252	7342	7413	7491	7586	7695	7798	7873	7912	7927	7935	7957	8001	8067	8142	8211	8261	8291	8306	8312
1237	1300	1436	1662	1995	2436	2946	3478	3992	4481	4943	5395	5853	6330	6791	7184	7459	7626	7724	7798	7873	7955	8042	8118	8167	8184	8175	8159	8154	8171	8207	8253	8295	8324	8340	8347	8349
1202	1231	1306	1464	1750	2186	2741	3347	3936	4479	4971	5449	5944	6479	7007	7458	7776	7971	8090	8180	8262	8342	8415	8475	8513	8522	8507	8480	8455	8441	8437	8440	8445	8449	8452	8455	8456
1190	1201	1241	1352	1596	2011	2581	3228	3864	4433	4934	5417	5937	6519	7109	7622	7991	8229	8386	8510	8617	8709	8786	8849	8893	8913	8909	8886	8852	8813	8771	8731	8700	8683	8679	8682	8686
1206	1210	1232	1315	1525	1912	2472	3125	3766	4325	4800	5260	5780	6391	7029	7599	8027	8326	8544	8725	8879	9011	9123	9217	9291	9339	9358	9351	9320	9267	9199	9130	9076	9050	9051	9065	9078
1254	1255	1271	1337	1521	1876	2406	3029	3634	4142	4560	4972	5468	6083	6745	7354	7836	8203	8498	8757	8985	9189	9373	9537	9669	9762	9815	9833	9815	9764	9688	9608	9551	9532	9551	9586	9612
1341	1341	1354	1410	1573	1892	2372	2935	3472	3908	4257	4611	5065	5654	6306	6922	7434	7860	8233	8582	8907	9214	9508	9775	9992	10147	10243	10291	10296	10261	10197	10129	10085	10084	10124	10180	10220
1462	1463	1474	1524	1666	1944	2358	2841	3297	3663	3956	4264	4675	5219	5829	6416	6925	7379	7813	8245	8677	9109	9537	9932	10253	10478	10620	10700	10733	10727	10691	10650	10628	10643	10694	10756	10798
1607	1608	1618	1663	1784	2017	2360	2760	3141	3458	3723	4009	4388	4882	5434	5968	6444	6894	7359	7860	8394	8958	9529	10057	10481	10773	10953	11060	11121	11147	11149	11142	11143	11163	11201	11243	11272
1756	1757	1767	1806	1907	2097	2375	2703	3028	3318	3580	3866	4229	4686	5185	5663	6095	6520	6991	7537	8160	8844	9549	10196	10707	11052	11262	11389	11472	11529	11567	11592	11606	11611	11611	11609	11607
1897	1898	1907	1940	2024	2179	2405	2680	2967	3245	3515	3811	4170	4604	5067	5505	5899	6297	6763	7338	8029	8812	9624	10363	10939	11322	11554	11698	11800	11882	11947	11992	12006	11983	11930	11871	11831
2018	2019	2027	2055	2125	2254	2443	2680	2942	3213	3490	3796	4157	4583	5029	5445	5817	6198	6660	7253	7992	8844	9731	10535	11156	11569	11822	11985	12106	12207	12289	12342	12345	12287	12181	12066	11992
2111	2112	2119	2144	2203	2313	2477	2688	2931	3197	3477	3789	4154	4579	5020	5430	5795	6170	6634	7241	8010	8906	9841	10688	11342	11782	12058	12242	12382	12498	12592	12646	12635	12546	12399	12247	12151
2167	2168	2174	2196	2249	2346	2492	2685	2916	3176	3458	3776	4145	4574	5018	5430	5797	6174	6643	7263	8052	8977	9942	10818	11497	11960	12259	12465	12623	12754	12857	12911	12890	12784	12618	12450	12347
2182	2183	2189	2208	2257	2345	2481	2663	2887	3146	3430	3752	4128	4562	5012	5429	5800	6183	6660	7294	8102	9050	10038	10933	11632	12114	12434	12660	12836	12980	13092	13150	13130	13023	12860	12698	12598
2158	2158	2164	2182	2228	2312	2441	2619	2842	3102	3391	3719	4099	4539	4993	5415	5792	6183	6672	7322	8151	9121	10129	11041	11753	12252	12591	12837	13030	13189	13311	13379	13369	13276	13130	12985	12897
2096	2097	2102	2121	2165	2249	2377	2556	2781	3046	3341	3674	4058	4499	4956	5381	5765	6167	6671	7337	8183	9170	10195	11125	11857	12377	12739	13008	13221	13396	13530	13611	13618	13547	13425	13303	13228
1996	1996	2001	2021	2068	2156	2291	2475	2706	2979	3281	3619	4005	4444	4898	5326	5722	6143	6664	7340	8187	9175	10212	11165	11929	12485	12881	13180	13420	13614	13765	13860	13885	13836	13738	13637	13575
1861	1859	1863	1883	1938	2038	2186	2383	2624	2904	3212	3552	3936	4369	4819	5253	5669	6119	6660	7334	8158	9122	10158	11141	11958	12568	13012	13352	13626	13847	14017	14129	14170	14141	14063	13979	13925
1718	1711	1706	1724	1787	1906	2076	2291	2540	2821	3125	3460	3835	4259	4704	5150	5596	6085	6652	7317	8102	9024	10050	11067	11945	12620	13121	13507	13821	14075	14272	14403	14462	14449	14386	14311	14263
1617	1597	1575	1583	1650	1789	1983	2212	2461	2728	3013	3329	3687	4098	4538	4994	5472	6004	6603	7267	8018	8901	9918	10966	11904	12639	13194	13624	13980	14271	14499	14655	14734	14736	14685	14619	14574
1596	1557	1509	1497	1563	1716	1928	2161	2392	2626	2873	3155	3489	3882	4313	4774	5272	5839	6472	7154	7902	8773	9793	10866	11848	12629	13224	13691	14084	14410	14670	14854	14957	14980	14944	14888	14849
1653	1595	1518	1484	1543	1702	1919	2141	2341	2528	2726	2968	3273	3646	4064	4515	5013	5593	6255	6970	7750	8642	9681	10777	11787	12596	13220	13715	14136	14492	14781	14993	15122	15170	15158	15120	15092
1746	1673	1575	1524	1577	1735	1947	2150	2314	2454	2605	2808	3085	3440	3842	4273	4752	5324	6002	6756	7584	8518	9587	10701	11728	12555	13198	13713	14158	14538	14851	15087	15242	15317	15333	15321	15307
1828	1748	1641	1583	1633	1789	1992	2177	2312	2417	2534	2707	2964	3304	3690	4099	4551	5102	5783	6566	7439	8414	9512	10643	11680	12517	13174	13707	14173	14575	14908	15162	15336	15435	15479	15493	15496
1877	1798	1691	1635	1686	1840	2036	2209	2328	2414	2510	2665	2907	3236	3612	4003	4433	4965	5642	6440	7343	8346	9464	10605	11649	12492	13159	13708	14192	14613	14963	15230	15415	15531	15597	15632	15648
1898	1821	1719	1670	1724	1877	2068	2236	2350	2429	2516	2660	2892	3215	3584	3966	4384	4903	5575	6379	7295	8312	9440	10586	11633	12480	13154	13713	14213	14650	15012	15285	15476	15601	15680	15730	15755
1904	1829	1732	1686	1743	1895	2086	2251	2365	2443	2526	2664	2892	3211	3578	3957	4370	4884	5554	6360	7280	8301	9432	10579	11628	12476	13153	13718	14226	14672	15040	15317	15510	15638	15723	15780	15810
]';
fc_o2_map=fliplr(fc_o2_map);

% (C), engine exhaust gas temperature indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_extmp_map=[437.73	435.82	432.47	428.58	425.15	422.96	422.26	423.22	425.82	430.13	436.1	444.36	455.51	469.56	484.57	498.11	508.77	517.33	525.26	533.24	540.97	547.93	554.54	561.92	572.23	588.06	612.41	647.76	694.41	747.73	795.87	823.03	817.88	781.94	730.3	682.81	655.34
440.05	437.86	433.99	429.51	425.53	422.95	422	422.74	425.07	429	434.58	442.54	453.58	467.7	482.9	496.71	507.65	516.49	524.68	532.87	540.74	547.8	554.65	562.9	575.76	596.5	627.53	668.62	716.82	765.72	805.19	823.26	812.24	774.75	724.61	679.49	653.59
443.39	440.79	436.21	430.85	426.04	422.78	421.32	421.61	423.41	426.7	431.61	439	449.68	463.68	478.94	492.96	504.29	513.66	522.45	531.2	539.52	546.92	554.31	564.13	581.11	609.53	650.87	700.58	750.5	791.41	815.93	818.61	798.01	758.73	712.31	672.38	649.82
446.26	443.42	438.35	432.32	426.71	422.59	420.3	419.79	420.81	423.29	427.39	433.99	443.94	457.26	472	485.8	497.36	507.43	517.14	526.82	535.93	544.05	552.42	564.31	585.85	622.19	673.63	731.27	781.55	812.41	819.46	804.64	773.7	733.85	693.73	661.71	644.17
448.12	445.35	440.33	434.13	427.96	422.85	419.28	417.47	417.36	418.87	422.06	427.66	436.4	448.3	461.68	474.6	486.07	496.81	507.61	518.45	528.63	537.86	547.74	562.19	588.2	631.2	690.2	752.86	801.56	822.32	813.35	783.1	743.72	704.95	672.54	649.61	637.77
449.26	446.92	442.56	436.8	430.41	424.19	418.89	415.2	413.47	413.71	415.81	420.17	427.28	437.14	448.47	459.91	470.9	482.09	493.83	505.71	516.94	527.53	539.42	556.98	587.38	635.71	699.53	764.34	810.15	822.46	801.46	760.3	715.59	678.92	653.75	638.93	632.12
450.13	448.45	445.14	440.25	433.93	426.61	419.28	413.22	409.38	408	408.79	411.74	416.98	424.48	433.36	442.93	453.03	464.2	476.37	488.86	500.99	513.12	527.47	548.63	583.39	636.07	702.69	767.57	810.08	816.44	788.38	741.66	694.97	660.71	640.8	631.52	628.09
450.88	449.88	447.69	443.83	437.79	429.57	420.21	411.59	405.39	402.21	401.68	403.28	406.71	411.81	418.13	425.54	434.25	444.68	456.46	468.89	481.56	495.25	512.25	537.05	575.73	631.66	699.5	763.21	802.62	805.45	774.65	726.9	681.11	649.42	632.84	626.58	624.88
451.3	450.81	449.5	446.55	440.9	432.05	420.99	410.06	401.66	396.96	395.47	396.16	398.15	401.09	404.88	409.9	416.63	425.41	435.81	447.34	460.01	474.91	494.14	521.82	563.15	620.66	688.14	749.73	786.33	787.4	756.95	711.51	668.96	640.32	626.01	621.12	620.06
450.99	450.8	450.05	447.76	442.5	433.39	421.17	408.6	398.72	393.25	391.59	392.09	393.16	394.3	395.73	398.23	402.54	408.97	417.15	426.93	438.75	453.83	474	502.71	544.44	601.17	666.6	725.4	759.75	760.44	732.26	691.08	653.27	628.24	615.94	611.75	610.82
449.12	449.06	448.58	446.74	441.9	432.93	420.38	407.22	396.98	391.8	390.85	391.86	392.6	392.39	391.77	391.93	393.76	397.55	403.08	410.47	420.49	434.37	453.59	480.97	520.57	574.23	636.19	691.86	724.55	725.82	700.9	664.73	632.11	610.77	600.21	596.31	595.22
444.64	444.62	444.29	442.78	438.5	430.24	418.42	406.05	396.79	392.87	393.25	395.14	395.89	394.74	392.54	390.78	390.48	391.86	394.77	399.53	407.08	418.61	435.35	459.69	495.48	544.7	602.38	654.73	686.06	688.29	666.67	635.07	606.93	588.58	579.22	575.29	573.88
436.48	436.48	436.25	435.1	431.64	424.81	414.95	404.79	397.68	395.64	397.43	400.12	400.94	399.22	395.99	392.92	391.02	390.41	390.94	393.1	398	406.84	420.75	441.91	474.04	519.11	572.59	621.45	650.9	653.34	633.89	605.45	580.31	563.84	555.15	551.09	549.42
424.67	424.68	424.56	423.84	421.5	416.77	409.9	403.02	398.77	398.65	401.43	404.43	405.14	403.06	399.33	395.54	392.57	390.37	388.82	388.61	391.12	397.72	409.57	428.79	458.78	501.13	551.14	596.31	622.82	623.98	604.93	577.91	554.33	539.05	531.09	527.42	525.94
410.2	410.22	410.22	409.98	408.95	406.68	403.29	400.1	398.75	400.15	403.24	405.91	406.23	403.91	400.02	395.95	392.3	388.84	385.49	383.28	383.95	389.14	400.06	418.78	448.24	489.37	536.8	578.37	601.19	599.78	579.67	552.82	530.09	516.03	509.49	507.19	506.56
394.97	395	395.11	395.34	395.58	395.67	395.56	395.72	396.78	399.05	401.79	403.6	403.25	400.63	396.69	392.52	388.42	384.01	379.29	375.6	375.09	379.67	390.51	409.53	439.09	479.55	524.89	563.21	582.46	578.33	556.86	530	508.3	496.08	491.93	491.95	492.79
380.39	380.43	380.64	381.26	382.54	384.55	387.05	389.76	392.44	394.98	396.94	397.63	396.42	393.37	389.29	385.01	380.6	375.57	369.96	365.43	364.39	368.94	380.05	399.38	428.8	468.32	511.7	547.35	563.95	557.97	535.85	509.56	489.45	479.66	478.44	481.18	483.71
367.23	367.27	367.55	368.47	370.51	373.87	378.17	382.53	386.1	388.45	389.45	388.96	386.85	383.28	378.95	374.52	369.92	364.6	358.64	353.86	352.78	357.54	368.89	388.17	416.93	455.03	496.41	529.95	544.91	538.22	516.45	491.44	473.41	466.27	467.95	473.26	477.3
355.41	355.46	355.79	356.91	359.44	363.69	369.12	374.44	378.37	380.27	380.28	378.7	375.77	371.72	367.18	362.65	358.03	352.78	346.96	342.38	341.55	346.47	357.76	376.54	404.21	440.67	480.12	511.98	525.99	519.35	498.53	475.19	459.28	454.56	458.6	465.82	470.93
344.71	344.77	345.13	346.36	349.2	353.98	360.1	365.94	369.93	371.29	370.38	367.88	364.3	359.95	355.33	350.87	346.45	341.55	336.25	332.19	331.68	336.56	347.37	365.22	391.51	426.22	463.82	494.21	507.57	501.25	481.58	459.88	445.86	443.02	448.68	457.05	462.74
334.84	334.89	335.26	336.53	339.52	344.59	351.08	357.17	361.03	361.79	360.03	356.74	352.75	348.38	344	339.89	335.9	331.56	326.94	323.5	323.27	327.91	337.96	354.62	379.39	412.31	448.14	477.14	489.91	483.91	465.25	444.94	432.32	430.67	437.15	445.96	451.81
325.28	325.33	325.69	326.94	329.93	335.07	341.66	347.74	351.33	351.46	348.92	345.02	340.87	336.82	333.05	329.59	326.18	322.4	318.34	315.38	315.37	319.81	329.24	344.86	368.24	399.48	433.61	461.29	473.45	467.63	449.71	430.32	418.49	417.34	423.98	432.73	438.47
315.3	315.34	315.68	316.89	319.81	324.86	331.35	337.22	340.38	339.87	336.63	332.31	328.28	324.88	322.05	319.46	316.67	313.24	309.4	306.61	306.73	311.22	320.54	335.74	358.24	388.2	420.96	447.59	459.26	453.4	435.65	416.41	404.57	403.2	409.46	417.81	423.3
303.39	303.5	303.93	305.23	308.2	313.21	319.54	325.15	327.97	327.06	323.51	319.15	315.5	312.87	310.96	309.14	306.74	303.34	299.33	296.39	296.61	301.49	311.39	326.93	349.25	378.52	410.55	436.73	448.22	442.05	423.68	403.51	390.68	388.4	393.91	401.72	406.91
287.8	288.2	289.14	291.02	294.44	299.66	305.98	311.51	314.3	313.53	310.29	306.41	303.39	301.49	300.27	298.88	296.56	292.89	288.47	285.21	285.49	290.9	301.7	318	340.66	369.85	402	428.57	440.39	433.72	413.85	391.58	376.76	372.95	377.48	384.71	389.63
266.19	267.44	269.83	273.26	277.85	283.6	290.01	295.63	298.86	298.96	296.85	294.12	292.09	290.9	290.07	288.77	286.29	282.31	277.56	274.06	274.37	280.17	291.67	308.58	331.65	361.17	394.2	422.05	434.83	427.65	405.71	380.49	362.94	357.21	360.66	367.38	372.13
238	240.83	245.89	252.09	258.6	265.02	271.35	276.97	280.91	282.43	282.14	281.2	280.6	280.33	279.93	278.71	276.15	272.04	267.19	263.56	263.74	269.4	280.85	297.69	320.87	351.02	385.71	415.79	430.29	422.98	398.92	370.45	349.83	341.91	344.16	350.35	354.93
204.87	209.85	218.48	228.33	237.22	244.25	250.16	255.51	260.18	263.42	265.38	266.69	267.99	269.01	269.31	268.38	265.97	261.98	257.19	253.4	253.1	257.96	268.46	284.4	307.31	338.37	375.48	408.78	425.88	419.1	393.31	361.73	337.96	327.68	328.54	334.07	338.4
172.07	179.09	191.1	204.36	215.46	222.99	228.43	233.5	238.88	243.78	248.01	251.68	255.08	257.63	258.87	258.46	256.34	252.5	247.59	243.28	241.97	245.33	254.07	268.47	290.87	323.15	363.38	400.65	420.96	415.22	388.14	353.79	327.05	314.4	313.67	318.26	322.15
144.89	153.07	167.05	182.36	194.9	203.01	208.53	213.83	220.03	226.43	232.64	238.48	243.85	247.88	250.07	250.17	248.29	244.39	239.02	233.8	231.08	232.62	239.3	251.95	273.8	307.42	350.85	391.97	415.16	410.17	381.8	345.09	315.97	301.41	299.28	302.86	306.19
126.77	134.89	148.83	164.28	177.24	186.11	192.58	198.9	206.13	213.75	221.49	229.08	236.08	241.33	244.24	244.68	242.83	238.64	232.55	226.22	222.04	221.86	226.68	237.81	259.19	293.96	339.9	383.69	408.35	402.98	373.01	334.49	304.06	288.57	285.58	288.3	291.06
116.85	123.99	136.37	150.51	163.14	172.9	180.93	188.79	197.1	205.67	214.51	223.46	231.73	237.9	241.28	241.89	239.95	235.41	228.67	221.43	216.16	214.77	218.33	228.45	249.54	284.99	332.29	377.01	401.34	394.29	362.45	322.8	292.27	276.95	273.83	276.02	278.31
112.52	118.41	128.82	141.24	153.25	163.81	173.38	182.7	191.89	201.09	210.66	220.6	229.78	236.59	240.25	240.93	238.88	234.08	226.91	219.14	213.27	211.24	214.17	223.78	244.72	280.44	328.1	372.5	395.46	386.28	352.76	312.63	282.74	268.2	265.37	267.28	269.24
110.97	116.05	125.17	136.46	148.04	159.07	169.57	179.74	189.42	198.92	208.88	219.36	229.06	236.21	240.02	240.72	238.62	233.71	226.38	218.4	212.32	210.06	212.78	222.22	243.11	278.88	326.49	370.34	392.14	381.5	347.01	306.75	277.48	263.6	261.06	262.87	264.66]'; % in deg. C
fc_extmp_map=fliplr(fc_extmp_map);

% (g/s), engine out total exhaust flow indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_exflow_map=[33.41	34.76	37.14	39.89	42.31	44	45.12	46.34	48.3	51.21	54.71	58.32	61.74	64.97	67.9	70.43	72.5	74.36	76.29	78.39	80.62	82.91	85.27	87.56	89.57	91.13	92.35	93.49	94.79	96.32	97.96	99.6	101.19	102.78	104.25	105.42	106.08
31.73	33.26	35.95	39.07	41.83	43.75	44.99	46.25	48.19	51.06	54.51	58.09	61.52	64.79	67.78	70.38	72.51	74.41	76.36	78.46	80.68	82.96	85.31	87.62	89.68	91.36	92.76	94.06	95.46	96.97	98.5	100.02	101.51	103	104.39	105.5	106.12
29.29	31.04	34.13	37.72	40.92	43.17	44.62	45.97	47.92	50.74	54.12	57.66	61.1	64.45	67.56	70.28	72.53	74.51	76.5	78.61	80.81	83.07	85.42	87.74	89.88	91.74	93.39	94.96	96.51	98	99.42	100.78	102.12	103.46	104.69	105.67	106.21
27.15	28.97	32.22	36.04	39.52	42.08	43.82	45.37	47.42	50.24	53.57	57.09	60.56	64	67.26	70.15	72.55	74.65	76.71	78.83	81.01	83.25	85.58	87.9	90.11	92.13	94.03	95.85	97.56	99.09	100.47	101.74	102.96	104.12	105.15	105.92	106.34
25.72	27.4	30.4	34.03	37.51	40.3	42.43	44.37	46.67	49.58	52.9	56.41	59.92	63.48	66.91	69.98	72.56	74.8	76.93	79.07	81.23	83.45	85.76	88.08	90.33	92.46	94.53	96.52	98.36	99.96	101.39	102.67	103.85	104.86	105.66	106.21	106.49
24.87	26.19	28.6	31.67	34.89	37.86	40.48	42.98	45.68	48.77	52.13	55.65	59.22	62.91	66.51	69.78	72.53	74.91	77.12	79.29	81.44	83.63	85.93	88.25	90.52	92.72	94.88	96.95	98.85	100.53	102.04	103.41	104.59	105.5	106.11	106.47	106.63
24.29	25.17	26.85	29.19	32.01	35.08	38.21	41.34	44.52	47.85	51.28	54.82	58.46	62.28	66.05	69.51	72.42	74.93	77.2	79.39	81.55	83.76	86.05	88.39	90.69	92.93	95.12	97.21	99.12	100.83	102.41	103.86	105.07	105.92	106.41	106.63	106.71
23.86	24.35	25.36	27.01	29.38	32.45	35.95	39.63	43.27	46.86	50.39	53.96	57.66	61.62	65.57	69.19	72.23	74.81	77.13	79.36	81.55	83.78	86.11	88.47	90.8	93.08	95.29	97.37	99.25	100.96	102.57	104.06	105.29	106.11	106.54	106.7	106.74
23.58	23.8	24.34	25.45	27.4	30.31	33.96	37.98	41.98	45.8	49.43	53.04	56.82	60.91	65.02	68.79	71.92	74.54	76.89	79.17	81.42	83.71	86.07	88.46	90.82	93.13	95.35	97.42	99.29	100.99	102.62	104.12	105.35	106.15	106.54	106.67	106.7
23.43	23.51	23.78	24.53	26.11	28.75	32.35	36.5	40.69	44.65	48.31	51.91	55.73	59.94	64.22	68.15	71.39	74.09	76.51	78.85	81.17	83.52	85.91	88.32	90.69	93	95.23	97.32	99.22	100.96	102.61	104.12	105.32	106.08	106.45	106.56	106.58
23.34	23.37	23.51	24.04	25.33	27.7	31.12	35.21	39.4	43.31	46.85	50.33	54.11	58.39	62.82	66.92	70.34	73.22	75.81	78.3	80.73	83.14	85.58	88.01	90.38	92.68	94.92	97.05	99.02	100.84	102.55	104.06	105.22	105.93	106.26	106.36	106.37
23.23	23.24	23.34	23.74	24.84	26.97	30.17	34.05	38.04	41.69	44.95	48.17	51.8	56.06	60.56	64.8	68.43	71.57	74.45	77.19	79.82	82.39	84.96	87.5	89.93	92.26	94.52	96.68	98.72	100.63	102.38	103.89	105.02	105.69	105.99	106.07	106.08
23.06	23.07	23.14	23.49	24.47	26.41	29.35	32.92	36.56	39.84	42.73	45.64	49.03	53.12	57.53	61.78	65.52	68.91	72.11	75.21	78.18	81.07	83.94	86.75	89.39	91.84	94.14	96.33	98.4	100.33	102.1	103.6	104.71	105.36	105.64	105.72	105.73
22.82	22.82	22.9	23.22	24.12	25.87	28.51	31.71	34.96	37.88	40.47	43.13	46.27	50.1	54.26	58.31	61.97	65.44	68.87	72.33	75.74	79.12	82.5	85.79	88.81	91.49	93.88	96.07	98.11	100.01	101.75	103.22	104.31	104.95	105.25	105.35	105.37
22.53	22.54	22.61	22.92	23.75	25.31	27.64	30.45	33.35	36.03	38.48	41.01	43.98	47.52	51.33	55.03	58.43	61.76	65.27	69	72.88	76.86	80.89	84.76	88.23	91.18	93.69	95.88	97.86	99.69	101.36	102.79	103.85	104.51	104.84	104.98	105.04
22.22	22.23	22.31	22.61	23.35	24.74	26.77	29.25	31.87	34.42	36.85	39.37	42.23	45.55	49.04	52.37	55.42	58.51	61.93	65.83	70.12	74.69	79.35	83.78	87.65	90.84	93.44	95.64	97.57	99.34	100.95	102.32	103.36	104.02	104.41	104.62	104.73
21.92	21.93	22.01	22.28	22.96	24.18	25.97	28.19	30.61	33.08	35.52	38.06	40.86	44.02	47.27	50.31	53.08	55.92	59.22	63.21	67.82	72.87	78.05	82.9	87.04	90.37	93.03	95.24	97.16	98.89	100.46	101.79	102.79	103.48	103.92	104.21	104.37
21.61	21.62	21.69	21.94	22.55	23.65	25.25	27.27	29.53	31.92	34.33	36.86	39.61	42.67	45.77	48.65	51.23	53.9	57.11	61.13	65.96	71.35	76.88	82.01	86.31	89.72	92.41	94.64	96.56	98.29	99.82	101.13	102.12	102.83	103.33	103.69	103.89
21.23	21.24	21.31	21.54	22.1	23.11	24.58	26.44	28.57	30.85	33.2	35.67	38.37	41.35	44.37	47.16	49.65	52.23	55.39	59.44	64.39	69.98	75.72	81.02	85.43	88.89	91.61	93.86	95.8	97.52	99.03	100.32	101.32	102.06	102.61	103.02	103.25
20.76	20.77	20.83	21.05	21.57	22.51	23.89	25.64	27.65	29.83	32.09	34.49	37.13	40.06	43.04	45.79	48.24	50.77	53.9	57.95	62.96	68.66	74.53	79.95	84.43	87.94	90.7	92.97	94.91	96.62	98.12	99.4	100.41	101.19	101.78	102.22	102.47
20.16	20.17	20.23	20.44	20.93	21.82	23.13	24.8	26.72	28.81	30.98	33.3	35.89	38.8	41.77	44.5	46.94	49.46	52.57	56.62	61.66	67.43	73.39	78.88	83.41	86.95	89.73	92.01	93.96	95.67	97.16	98.43	99.46	100.27	100.9	101.37	101.63
19.44	19.44	19.5	19.69	20.16	21.01	22.26	23.87	25.72	27.72	29.8	32.06	34.61	37.5	40.47	43.22	45.66	48.18	51.29	55.36	60.45	66.29	72.33	77.86	82.42	85.96	88.74	91.04	93	94.71	96.2	97.49	98.54	99.39	100.06	100.55	100.83
18.56	18.56	18.61	18.79	19.24	20.05	21.25	22.8	24.58	26.5	28.51	30.71	33.22	36.09	39.06	41.82	44.29	46.83	49.97	54.06	59.18	65.08	71.17	76.77	81.35	84.91	87.71	90.03	92.02	93.77	95.29	96.6	97.69	98.57	99.27	99.79	100.08
17.45	17.46	17.51	17.69	18.12	18.91	20.07	21.57	23.29	25.17	27.13	29.29	31.75	34.58	37.52	40.28	42.79	45.39	48.57	52.66	57.73	63.6	69.71	75.37	80.05	83.68	86.55	88.94	91.02	92.84	94.43	95.79	96.91	97.81	98.53	99.06	99.36
16.1	16.1	16.16	16.35	16.8	17.61	18.77	20.25	21.95	23.79	25.74	27.87	30.28	33.03	35.9	38.65	41.24	43.97	47.22	51.23	56.11	61.77	67.79	73.52	78.37	82.18	85.2	87.74	89.96	91.91	93.59	95	96.15	97.06	97.78	98.31	98.6
14.55	14.56	14.61	14.83	15.34	16.21	17.42	18.91	20.59	22.42	24.34	26.43	28.76	31.41	34.18	36.94	39.65	42.58	45.94	49.87	54.45	59.75	65.6	71.36	76.42	80.47	83.7	86.43	88.83	90.94	92.73	94.21	95.38	96.29	97	97.51	97.8
13.06	13.05	13.1	13.35	13.92	14.86	16.13	17.63	19.27	21.01	22.83	24.82	27.06	29.59	32.28	35.04	37.9	41.06	44.58	48.45	52.76	57.71	63.34	69.11	74.35	78.63	82.08	85	87.59	89.85	91.76	93.32	94.54	95.46	96.15	96.64	96.91
11.84	11.82	11.85	12.11	12.72	13.72	15.01	16.46	17.96	19.51	21.14	22.97	25.07	27.48	30.09	32.84	35.79	39.13	42.83	46.74	50.94	55.69	61.19	66.96	72.32	76.75	80.37	83.43	86.16	88.55	90.59	92.25	93.54	94.5	95.2	95.68	95.94
11.06	11.01	11.01	11.26	11.89	12.89	14.15	15.48	16.78	18.06	19.43	21.04	23	25.31	27.83	30.5	33.43	36.8	40.59	44.61	48.87	53.63	59.11	64.89	70.32	74.84	78.56	81.72	84.55	87.04	89.19	90.96	92.34	93.36	94.09	94.57	94.83
10.66	10.57	10.53	10.76	11.38	12.37	13.57	14.76	15.84	16.86	17.97	19.39	21.22	23.44	25.87	28.41	31.18	34.42	38.2	42.32	46.74	51.64	57.2	63.02	68.48	73.05	76.83	80.05	82.95	85.5	87.72	89.55	91.01	92.08	92.85	93.35	93.61
10.49	10.37	10.29	10.48	11.09	12.07	13.23	14.32	15.24	16.07	17.01	18.29	20.04	22.2	24.55	26.93	29.49	32.52	36.19	40.35	44.92	49.97	55.64	61.5	66.99	71.58	75.4	78.67	81.62	84.22	86.46	88.29	89.74	90.82	91.6	92.11	92.38
10.43	10.27	10.15	10.31	10.9	11.89	13.04	14.1	14.95	15.68	16.53	17.74	19.45	21.58	23.87	26.14	28.52	31.37	34.92	39.06	43.72	48.87	54.62	60.52	66.01	70.62	74.47	77.79	80.79	83.42	85.63	87.4	88.77	89.79	90.54	91.05	91.33
10.39	10.22	10.06	10.19	10.78	11.78	12.95	14.01	14.85	15.56	16.37	17.56	19.25	21.37	23.63	25.84	28.12	30.85	34.32	38.44	43.12	48.32	54.1	60.02	65.52	70.14	74.01	77.38	80.42	83.06	85.24	86.92	88.18	89.12	89.82	90.31	90.57
10.37	10.19	10.01	10.12	10.7	11.72	12.9	13.98	14.82	15.53	16.34	17.53	19.21	21.32	23.58	25.76	28.01	30.69	34.13	38.23	42.93	48.14	53.93	59.86	65.36	69.98	73.87	77.25	80.32	82.97	85.12	86.75	87.94	88.81	89.47	89.94	90.19]'; % in cubic m/s
fc_exflow_map=fliplr(fc_exflow_map);

% (m^3/s), engine out total volume exhaust flow indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_exflow_vol_map=[0.066	0.068	0.073	0.078	0.082	0.085	0.088	0.09	0.094	0.101	0.109	0.117	0.126	0.135	0.144	0.152	0.158	0.164	0.17	0.176	0.183	0.19	0.197	0.204	0.211	0.219	0.229	0.241	0.258	0.276	0.293	0.305	0.307	0.301	0.29	0.28	0.274
0.063	0.065	0.07	0.076	0.081	0.085	0.087	0.09	0.094	0.1	0.108	0.117	0.125	0.135	0.144	0.152	0.158	0.164	0.17	0.177	0.183	0.19	0.197	0.204	0.213	0.222	0.234	0.249	0.265	0.282	0.297	0.306	0.306	0.299	0.289	0.279	0.273
0.058	0.061	0.067	0.074	0.08	0.084	0.087	0.089	0.093	0.099	0.107	0.115	0.124	0.133	0.142	0.151	0.158	0.164	0.17	0.177	0.183	0.19	0.197	0.205	0.215	0.227	0.242	0.26	0.277	0.292	0.302	0.306	0.304	0.296	0.286	0.277	0.272
0.054	0.057	0.063	0.07	0.077	0.082	0.085	0.088	0.092	0.098	0.105	0.113	0.122	0.131	0.141	0.149	0.156	0.163	0.169	0.176	0.183	0.19	0.197	0.206	0.217	0.231	0.25	0.27	0.288	0.301	0.306	0.305	0.299	0.291	0.282	0.275	0.271
0.051	0.054	0.06	0.067	0.073	0.078	0.082	0.086	0.09	0.096	0.103	0.111	0.119	0.129	0.138	0.147	0.154	0.161	0.168	0.175	0.182	0.189	0.197	0.206	0.218	0.235	0.256	0.278	0.296	0.305	0.306	0.301	0.293	0.284	0.277	0.272	0.27
0.05	0.052	0.057	0.062	0.068	0.073	0.078	0.083	0.088	0.094	0.101	0.108	0.116	0.126	0.135	0.143	0.151	0.158	0.166	0.173	0.18	0.187	0.195	0.205	0.218	0.237	0.259	0.282	0.299	0.307	0.305	0.296	0.287	0.279	0.273	0.27	0.268
0.049	0.05	0.053	0.058	0.063	0.068	0.073	0.079	0.085	0.091	0.098	0.105	0.113	0.122	0.131	0.14	0.148	0.155	0.162	0.169	0.177	0.184	0.193	0.204	0.218	0.238	0.261	0.284	0.3	0.306	0.302	0.292	0.282	0.275	0.27	0.268	0.267
0.048	0.049	0.051	0.053	0.058	0.063	0.069	0.075	0.082	0.089	0.095	0.102	0.11	0.119	0.127	0.136	0.143	0.151	0.158	0.165	0.173	0.18	0.19	0.201	0.217	0.237	0.261	0.283	0.298	0.303	0.298	0.289	0.279	0.272	0.268	0.267	0.267
0.047	0.048	0.049	0.051	0.054	0.059	0.065	0.072	0.079	0.086	0.093	0.1	0.107	0.115	0.124	0.132	0.139	0.146	0.153	0.16	0.168	0.176	0.185	0.198	0.214	0.235	0.258	0.28	0.294	0.298	0.294	0.285	0.276	0.27	0.266	0.265	0.265
0.047	0.047	0.048	0.049	0.052	0.056	0.062	0.069	0.076	0.083	0.09	0.097	0.104	0.112	0.121	0.128	0.136	0.142	0.148	0.155	0.162	0.171	0.181	0.193	0.209	0.229	0.252	0.273	0.287	0.291	0.287	0.279	0.271	0.266	0.263	0.263	0.262
0.047	0.047	0.047	0.048	0.05	0.054	0.059	0.066	0.073	0.08	0.087	0.094	0.101	0.109	0.117	0.125	0.132	0.138	0.144	0.151	0.158	0.166	0.175	0.187	0.203	0.222	0.244	0.264	0.277	0.281	0.278	0.271	0.265	0.261	0.259	0.258	0.257
0.046	0.046	0.046	0.047	0.049	0.052	0.058	0.064	0.071	0.078	0.084	0.09	0.097	0.105	0.113	0.121	0.127	0.134	0.14	0.146	0.153	0.161	0.17	0.181	0.196	0.214	0.234	0.253	0.266	0.27	0.268	0.263	0.257	0.254	0.252	0.251	0.251
0.045	0.046	0.046	0.046	0.048	0.051	0.056	0.062	0.068	0.074	0.08	0.086	0.092	0.1	0.108	0.115	0.122	0.128	0.134	0.141	0.148	0.155	0.165	0.175	0.189	0.206	0.225	0.243	0.255	0.26	0.258	0.254	0.249	0.246	0.244	0.243	0.243
0.044	0.044	0.044	0.045	0.047	0.049	0.054	0.06	0.065	0.071	0.076	0.082	0.088	0.095	0.102	0.109	0.115	0.122	0.128	0.135	0.142	0.15	0.159	0.171	0.184	0.201	0.219	0.236	0.247	0.251	0.25	0.245	0.241	0.238	0.237	0.236	0.236
0.043	0.043	0.043	0.044	0.045	0.048	0.052	0.057	0.063	0.068	0.073	0.078	0.083	0.09	0.097	0.103	0.109	0.115	0.121	0.127	0.135	0.144	0.154	0.166	0.181	0.197	0.215	0.231	0.241	0.244	0.242	0.237	0.233	0.231	0.23	0.229	0.229
0.041	0.041	0.042	0.042	0.044	0.046	0.05	0.055	0.06	0.065	0.07	0.075	0.08	0.086	0.092	0.098	0.103	0.108	0.113	0.12	0.128	0.138	0.149	0.162	0.177	0.194	0.212	0.226	0.235	0.237	0.235	0.23	0.226	0.224	0.224	0.224	0.225
0.04	0.04	0.04	0.041	0.042	0.045	0.048	0.052	0.057	0.062	0.067	0.072	0.077	0.082	0.088	0.093	0.097	0.102	0.107	0.114	0.122	0.133	0.145	0.159	0.174	0.191	0.207	0.221	0.229	0.231	0.228	0.223	0.22	0.218	0.219	0.22	0.221
0.039	0.039	0.039	0.039	0.041	0.043	0.046	0.05	0.055	0.059	0.064	0.068	0.073	0.079	0.084	0.088	0.093	0.097	0.101	0.108	0.117	0.128	0.14	0.154	0.17	0.186	0.202	0.215	0.223	0.224	0.221	0.217	0.214	0.213	0.215	0.217	0.219
0.037	0.037	0.038	0.038	0.039	0.041	0.044	0.048	0.052	0.057	0.061	0.065	0.07	0.075	0.08	0.084	0.088	0.092	0.097	0.103	0.112	0.123	0.136	0.15	0.165	0.181	0.196	0.209	0.216	0.218	0.215	0.211	0.209	0.209	0.211	0.214	0.216
0.036	0.036	0.036	0.037	0.038	0.04	0.043	0.046	0.05	0.054	0.058	0.062	0.066	0.071	0.076	0.08	0.084	0.088	0.092	0.099	0.108	0.119	0.132	0.146	0.16	0.176	0.19	0.203	0.21	0.211	0.209	0.205	0.203	0.204	0.207	0.21	0.212
0.034	0.034	0.035	0.035	0.036	0.038	0.041	0.044	0.048	0.051	0.055	0.059	0.063	0.068	0.073	0.077	0.081	0.084	0.089	0.095	0.104	0.115	0.128	0.141	0.155	0.17	0.185	0.196	0.203	0.205	0.202	0.199	0.198	0.199	0.202	0.205	0.207
0.033	0.033	0.033	0.033	0.034	0.036	0.039	0.042	0.045	0.049	0.052	0.056	0.06	0.064	0.069	0.074	0.077	0.081	0.086	0.092	0.101	0.112	0.124	0.137	0.151	0.165	0.179	0.19	0.197	0.198	0.196	0.193	0.192	0.194	0.197	0.2	0.202
0.031	0.031	0.031	0.031	0.032	0.034	0.036	0.039	0.043	0.046	0.049	0.052	0.056	0.061	0.066	0.07	0.074	0.078	0.082	0.089	0.098	0.108	0.121	0.134	0.147	0.16	0.174	0.185	0.191	0.193	0.191	0.188	0.187	0.188	0.192	0.195	0.197
0.028	0.028	0.029	0.029	0.03	0.031	0.034	0.037	0.04	0.043	0.046	0.049	0.053	0.057	0.062	0.066	0.07	0.074	0.079	0.085	0.094	0.104	0.117	0.129	0.143	0.156	0.169	0.18	0.186	0.188	0.186	0.183	0.182	0.183	0.186	0.189	0.191
0.026	0.026	0.026	0.026	0.027	0.029	0.031	0.034	0.037	0.04	0.042	0.046	0.049	0.054	0.058	0.063	0.067	0.07	0.075	0.081	0.089	0.1	0.112	0.125	0.138	0.151	0.164	0.175	0.182	0.184	0.182	0.178	0.177	0.178	0.18	0.183	0.185
0.022	0.022	0.023	0.023	0.024	0.026	0.028	0.031	0.034	0.036	0.039	0.043	0.046	0.05	0.055	0.059	0.063	0.067	0.072	0.077	0.085	0.095	0.106	0.119	0.133	0.146	0.16	0.171	0.179	0.18	0.178	0.174	0.172	0.172	0.174	0.177	0.179
0.019	0.019	0.02	0.02	0.021	0.023	0.025	0.028	0.03	0.033	0.036	0.039	0.043	0.047	0.051	0.055	0.059	0.063	0.068	0.074	0.081	0.09	0.101	0.114	0.127	0.141	0.155	0.167	0.175	0.177	0.174	0.17	0.167	0.166	0.168	0.171	0.173
0.016	0.016	0.017	0.017	0.019	0.02	0.022	0.025	0.027	0.03	0.033	0.035	0.039	0.043	0.047	0.051	0.055	0.059	0.064	0.07	0.077	0.085	0.095	0.108	0.121	0.135	0.15	0.163	0.172	0.174	0.171	0.165	0.162	0.161	0.163	0.165	0.166
0.014	0.014	0.015	0.015	0.017	0.018	0.02	0.022	0.025	0.027	0.029	0.032	0.035	0.038	0.042	0.046	0.05	0.055	0.06	0.065	0.072	0.08	0.09	0.101	0.115	0.129	0.144	0.158	0.167	0.17	0.167	0.161	0.157	0.156	0.157	0.159	0.16
0.013	0.013	0.013	0.014	0.015	0.017	0.019	0.021	0.022	0.024	0.026	0.028	0.031	0.035	0.039	0.042	0.046	0.051	0.055	0.061	0.067	0.075	0.084	0.096	0.108	0.123	0.139	0.153	0.163	0.166	0.162	0.156	0.152	0.15	0.151	0.153	0.154
0.012	0.012	0.012	0.013	0.014	0.016	0.018	0.019	0.021	0.022	0.024	0.026	0.029	0.033	0.036	0.04	0.043	0.047	0.052	0.057	0.063	0.071	0.08	0.091	0.103	0.118	0.134	0.149	0.159	0.162	0.158	0.152	0.147	0.145	0.145	0.147	0.148
0.012	0.012	0.012	0.012	0.014	0.015	0.017	0.019	0.02	0.021	0.023	0.025	0.028	0.031	0.035	0.038	0.041	0.045	0.049	0.055	0.061	0.068	0.077	0.087	0.1	0.114	0.131	0.146	0.156	0.158	0.154	0.147	0.142	0.14	0.141	0.142	0.143
0.011	0.011	0.012	0.012	0.013	0.015	0.017	0.018	0.02	0.021	0.023	0.025	0.028	0.031	0.034	0.038	0.041	0.044	0.048	0.053	0.059	0.067	0.075	0.086	0.098	0.113	0.129	0.144	0.154	0.155	0.151	0.144	0.139	0.137	0.138	0.139	0.14
0.011	0.011	0.011	0.012	0.013	0.015	0.016	0.018	0.02	0.021	0.022	0.025	0.027	0.031	0.034	0.038	0.041	0.044	0.048	0.053	0.059	0.066	0.075	0.085	0.097	0.112	0.128	0.143	0.153	0.154	0.149	0.142	0.137	0.135	0.136	0.137	0.138
]';
fc_exflow_vol_map=fliplr(fc_exflow_vol_map);

% adjust ex gas temperature map to estimate temperature drop across manifold & turbo 
%  assume delta T is equal to 10% of the temp diff between gas temp and 20 C ambient
fc_extmp_map=fc_extmp_map + (fc_extmp_map - 20)*0.1;

% create BS** maps for plotting purposes
[T,w]=meshgrid(fc_map_trq(2:end),fc_map_spd);
fc_map_kW=T.*w/1000;
fc_fuel_map_gpkWh=fc_fuel_map(:,2:end)./fc_map_kW*3600;
fc_co_map_gpkWh=fc_co_map(:,2:end)./fc_map_kW*3600;
fc_hc_map_gpkWh=fc_hc_map(:,2:end)./fc_map_kW*3600;
fc_nox_map_gpkWh=fc_nox_map(:,2:end)./fc_map_kW*3600;
fc_pm_map_gpkWh=fc_pm_map(:,2:end)./fc_map_kW*3600;

% replace zero torque column values
fc_fuel_map_gpkWh=[fc_fuel_map_gpkWh(:,1)*1.1 fc_fuel_map_gpkWh];
fc_co_map_gpkWh=[fc_co_map_gpkWh(:,1)*1.1 fc_co_map_gpkWh];
fc_nox_map_gpkWh=[fc_nox_map_gpkWh(:,1)*1.1 fc_nox_map_gpkWh];
fc_hc_map_gpkWh=[fc_hc_map_gpkWh(:,1)*1.1 fc_hc_map_gpkWh];
fc_pm_map_gpkWh=[fc_pm_map_gpkWh(:,1)*1.1 fc_pm_map_gpkWh];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Cold Engine Maps
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% For cold testing:
%This spreadsheet contains an engine map for a Volkswagen 1.9 liter turbocharged, direct-injection diesel engine.
%The map was developed through CIDI Benchmarking activities of the Office of Advanced Automotive Technology,
%of the U.S. Department of Energy.  The mapping exercise was carried out by staff members of the Advanced
%Propulsion Technology Center at Oak Ridge National Laboratory in 1999.
%
%The mapping was conducted on an engine that was force-cooled but operating normally.  The engine was equipped
%with an alternator, but otherwise without accessory loads.  An engine map suitable for use with the ADVISOR
%model was subsequently produced from this data and delivered to NREL for inclusion with the model.
%Force-cooling was accomplished by maintaining the engine coolant temperature at ~70 F.  Engine intake air
%was also maintained at ~70 F.  Ambient temperature during the test was not controlled, but was not higher than
%~90 F.
%
%Exhaust emissions tables are in units of mg/s, fuel consumption is in g/s, exhaust temperature is in degrees C.
%Exhaust flow has been reported both gravimetrically (g/s) and volumetrically (cubic meters per second).
%

fc_cold_tmp=[21.1]; %deg C
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COLD FUEL USE AND EMISSIONS MAPS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (g/s) , fuel use map indexed vertically by fc_map_spd and 
% horizontally by fc_map_trq
fc_fuel_map_cold=[
   1.26	1.33	1.46	1.61	1.74	1.83	1.89	1.94	2.02	2.14	2.28	2.42	2.53	2.62	2.7	2.78	2.87	3.01	3.19	3.39	3.57	3.71	3.82	3.89	3.94	3.99	4.03	4.08	4.14	4.2	4.27	4.33	4.38	4.41	4.43	4.44	4.45
   1.26	1.33	1.46	1.6	1.73	1.82	1.88	1.94	2.02	2.13	2.27	2.4	2.51	2.61	2.69	2.77	2.87	3.01	3.19	3.39	3.58	3.72	3.82	3.89	3.95	4	4.05	4.1	4.17	4.23	4.29	4.34	4.38	4.41	4.43	4.44	4.45
   1.26	1.32	1.44	1.58	1.71	1.8	1.86	1.92	2	2.11	2.24	2.36	2.47	2.57	2.66	2.75	2.86	3.01	3.19	3.4	3.58	3.72	3.82	3.89	3.95	4.01	4.08	4.14	4.21	4.27	4.32	4.36	4.39	4.42	4.43	4.44	4.45
   1.25	1.3	1.41	1.53	1.65	1.75	1.82	1.89	1.97	2.08	2.19	2.31	2.42	2.52	2.62	2.72	2.84	3	3.19	3.39	3.57	3.71	3.81	3.88	3.95	4.03	4.1	4.18	4.25	4.31	4.36	4.39	4.41	4.43	4.44	4.45	4.45
   1.23	1.28	1.36	1.46	1.57	1.67	1.76	1.84	1.93	2.03	2.14	2.25	2.36	2.46	2.56	2.67	2.8	2.97	3.17	3.37	3.54	3.68	3.78	3.86	3.94	4.02	4.11	4.2	4.28	4.34	4.38	4.4	4.42	4.43	4.44	4.45	4.45
   1.22	1.25	1.3	1.38	1.48	1.57	1.67	1.77	1.87	1.97	2.08	2.18	2.29	2.39	2.49	2.6	2.74	2.92	3.12	3.32	3.49	3.62	3.73	3.82	3.91	4	4.1	4.2	4.29	4.35	4.39	4.41	4.43	4.44	4.45	4.45	4.45
   1.2	1.22	1.25	1.31	1.39	1.48	1.58	1.69	1.79	1.9	2	2.11	2.2	2.3	2.4	2.52	2.66	2.84	3.04	3.24	3.41	3.54	3.65	3.75	3.85	3.96	4.07	4.18	4.27	4.33	4.38	4.41	4.43	4.44	4.45	4.45	4.46
   1.19	1.2	1.22	1.25	1.32	1.4	1.5	1.61	1.72	1.82	1.93	2.03	2.12	2.21	2.31	2.43	2.56	2.74	2.94	3.13	3.3	3.44	3.56	3.66	3.77	3.88	4	4.11	4.21	4.29	4.35	4.39	4.43	4.44	4.45	4.46	4.46
   1.18	1.19	1.2	1.22	1.27	1.34	1.44	1.54	1.64	1.75	1.85	1.95	2.04	2.13	2.23	2.33	2.46	2.63	2.82	3.01	3.18	3.32	3.45	3.56	3.68	3.79	3.91	4.02	4.12	4.21	4.29	4.36	4.4	4.43	4.45	4.45	4.45
   1.17	1.17	1.18	1.2	1.24	1.3	1.38	1.48	1.58	1.68	1.79	1.89	1.98	2.07	2.15	2.25	2.37	2.53	2.71	2.89	3.06	3.2	3.33	3.46	3.57	3.68	3.79	3.9	4	4.1	4.19	4.28	4.35	4.4	4.42	4.43	4.43
   1.16	1.16	1.16	1.18	1.21	1.26	1.34	1.42	1.52	1.63	1.73	1.83	1.92	2	2.09	2.18	2.29	2.44	2.61	2.78	2.95	3.09	3.22	3.34	3.46	3.56	3.66	3.75	3.84	3.94	4.06	4.17	4.26	4.32	4.35	4.35	4.36
   1.13	1.13	1.13	1.15	1.18	1.22	1.29	1.38	1.47	1.57	1.67	1.77	1.85	1.94	2.02	2.11	2.22	2.36	2.52	2.69	2.85	2.99	3.12	3.23	3.33	3.42	3.51	3.58	3.67	3.77	3.88	4	4.1	4.17	4.2	4.21	4.21
   1.09	1.09	1.09	1.11	1.13	1.18	1.25	1.33	1.42	1.51	1.61	1.7	1.78	1.86	1.94	2.03	2.14	2.28	2.44	2.61	2.76	2.9	3.01	3.11	3.2	3.28	3.35	3.42	3.49	3.58	3.69	3.8	3.89	3.96	3.99	4	4
   1.05	1.05	1.05	1.06	1.09	1.13	1.19	1.27	1.35	1.44	1.53	1.62	1.7	1.78	1.86	1.95	2.06	2.19	2.35	2.52	2.67	2.79	2.9	2.99	3.06	3.13	3.2	3.26	3.32	3.4	3.49	3.59	3.67	3.72	3.74	3.75	3.75
   1	1	1.01	1.02	1.04	1.08	1.13	1.2	1.29	1.37	1.46	1.54	1.62	1.7	1.78	1.87	1.97	2.1	2.26	2.42	2.56	2.67	2.77	2.85	2.92	2.98	3.04	3.1	3.17	3.25	3.33	3.41	3.47	3.51	3.53	3.53	3.54
   0.96	0.96	0.96	0.97	0.99	1.03	1.08	1.14	1.22	1.31	1.39	1.48	1.55	1.63	1.71	1.79	1.89	2.01	2.16	2.3	2.44	2.54	2.63	2.7	2.77	2.83	2.89	2.96	3.03	3.11	3.19	3.26	3.32	3.36	3.37	3.38	3.38
   0.92	0.92	0.92	0.93	0.95	0.98	1.03	1.09	1.17	1.25	1.33	1.42	1.49	1.56	1.64	1.71	1.8	1.91	2.05	2.18	2.31	2.41	2.49	2.57	2.63	2.7	2.76	2.83	2.91	2.99	3.07	3.15	3.21	3.25	3.26	3.27	3.27
   0.88	0.88	0.88	0.89	0.91	0.94	0.99	1.05	1.12	1.2	1.28	1.36	1.43	1.5	1.56	1.63	1.71	1.81	1.93	2.05	2.17	2.28	2.37	2.45	2.52	2.58	2.65	2.72	2.8	2.88	2.96	3.04	3.1	3.14	3.16	3.17	3.17
   0.84	0.84	0.85	0.86	0.87	0.91	0.95	1.01	1.08	1.15	1.22	1.3	1.36	1.42	1.49	1.55	1.61	1.7	1.8	1.92	2.04	2.15	2.25	2.34	2.42	2.49	2.56	2.63	2.7	2.78	2.85	2.92	2.98	3.02	3.04	3.04	3.04
   0.8	0.8	0.8	0.81	0.83	0.87	0.91	0.97	1.03	1.1	1.17	1.23	1.29	1.35	1.41	1.47	1.52	1.6	1.69	1.8	1.92	2.04	2.14	2.24	2.32	2.4	2.48	2.55	2.62	2.68	2.75	2.81	2.86	2.89	2.91	2.91	2.91
   0.76	0.76	0.76	0.77	0.79	0.82	0.86	0.92	0.98	1.04	1.1	1.16	1.22	1.28	1.34	1.39	1.44	1.51	1.6	1.7	1.81	1.92	2.03	2.13	2.23	2.31	2.39	2.47	2.53	2.59	2.65	2.7	2.75	2.78	2.79	2.79	2.79
   0.71	0.71	0.71	0.72	0.74	0.77	0.81	0.86	0.92	0.98	1.04	1.09	1.15	1.21	1.27	1.32	1.37	1.43	1.51	1.6	1.71	1.81	1.92	2.02	2.12	2.21	2.29	2.37	2.44	2.5	2.56	2.62	2.66	2.7	2.71	2.72	2.72
   0.66	0.66	0.66	0.67	0.68	0.71	0.75	0.8	0.86	0.91	0.96	1.02	1.07	1.13	1.19	1.25	1.3	1.36	1.43	1.51	1.61	1.7	1.8	1.9	1.99	2.09	2.18	2.26	2.33	2.4	2.48	2.55	2.61	2.65	2.67	2.68	2.68
   0.61	0.61	0.61	0.62	0.63	0.66	0.69	0.74	0.79	0.84	0.89	0.94	0.99	1.05	1.11	1.17	1.22	1.28	1.34	1.42	1.5	1.59	1.68	1.77	1.86	1.96	2.05	2.13	2.21	2.3	2.39	2.48	2.56	2.62	2.65	2.66	2.66
   0.55	0.55	0.56	0.56	0.58	0.6	0.64	0.68	0.73	0.77	0.82	0.87	0.92	0.98	1.04	1.09	1.15	1.2	1.26	1.33	1.4	1.48	1.56	1.65	1.74	1.83	1.92	2	2.09	2.18	2.29	2.4	2.5	2.57	2.62	2.64	2.65
   0.49	0.49	0.5	0.51	0.52	0.55	0.58	0.63	0.67	0.71	0.76	0.81	0.86	0.91	0.97	1.02	1.07	1.12	1.17	1.24	1.31	1.39	1.47	1.54	1.62	1.71	1.79	1.88	1.97	2.07	2.18	2.3	2.41	2.51	2.58	2.62	2.64
   0.42	0.43	0.44	0.45	0.47	0.5	0.53	0.57	0.62	0.66	0.71	0.75	0.8	0.85	0.9	0.95	0.99	1.04	1.09	1.15	1.23	1.3	1.38	1.46	1.53	1.61	1.69	1.78	1.87	1.96	2.07	2.18	2.3	2.42	2.52	2.59	2.62
   0.35	0.36	0.38	0.4	0.42	0.45	0.49	0.53	0.57	0.61	0.66	0.7	0.74	0.79	0.84	0.88	0.92	0.96	1.01	1.07	1.14	1.22	1.3	1.38	1.45	1.53	1.61	1.7	1.79	1.87	1.96	2.07	2.19	2.32	2.45	2.55	2.6
   0.29	0.3	0.32	0.35	0.38	0.42	0.45	0.49	0.53	0.57	0.61	0.65	0.69	0.73	0.77	0.81	0.85	0.89	0.93	0.99	1.06	1.13	1.21	1.29	1.37	1.45	1.54	1.64	1.73	1.81	1.88	1.97	2.08	2.23	2.38	2.5	2.57
   0.24	0.26	0.29	0.32	0.35	0.39	0.42	0.45	0.49	0.53	0.57	0.6	0.64	0.68	0.72	0.76	0.79	0.83	0.87	0.92	0.98	1.05	1.13	1.21	1.29	1.37	1.47	1.58	1.67	1.75	1.82	1.9	2.01	2.15	2.31	2.44	2.51
   0.22	0.23	0.26	0.29	0.33	0.36	0.39	0.43	0.46	0.5	0.54	0.57	0.61	0.65	0.69	0.72	0.76	0.79	0.82	0.87	0.93	0.99	1.06	1.14	1.21	1.3	1.4	1.5	1.61	1.69	1.76	1.84	1.95	2.08	2.22	2.34	2.41
   0.2	0.22	0.24	0.27	0.31	0.34	0.38	0.41	0.45	0.48	0.52	0.56	0.59	0.63	0.67	0.71	0.74	0.77	0.8	0.85	0.9	0.95	1.02	1.09	1.16	1.24	1.33	1.43	1.53	1.62	1.71	1.8	1.89	2.01	2.12	2.21	2.27
   0.19	0.21	0.23	0.26	0.29	0.33	0.36	0.4	0.43	0.47	0.51	0.55	0.59	0.63	0.67	0.7	0.73	0.76	0.8	0.84	0.88	0.94	1	1.06	1.13	1.2	1.27	1.35	1.45	1.55	1.66	1.75	1.85	1.94	2.02	2.09	2.12
   0.19	0.2	0.23	0.25	0.28	0.32	0.35	0.39	0.43	0.47	0.51	0.55	0.59	0.63	0.66	0.7	0.73	0.76	0.79	0.83	0.88	0.93	0.99	1.05	1.12	1.17	1.23	1.31	1.4	1.51	1.62	1.73	1.82	1.89	1.96	2	2.03
]';
fc_fuel_map_cold=fliplr(fc_fuel_map_cold);

% (g/s), engine out HC emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_hc_map_cold=1e-3*[
3.82	4.61	6	7.61	9.04	10	10.5	10.72	10.87	11	10.95	10.41	9.09	7.01	4.99	4.72	8.79	18.94	35.12	54.55	74.39	92.7	109.35	123.28	132.43	134.22	127.72	113.24	92.35	67.45	42.69	22.35	9.23	2.78	0.66	0.26	0.28
3.8	4.57	5.94	7.53	8.95	9.91	10.4	10.56	10.57	10.46	10.17	9.46	8.15	6.25	4.54	4.67	9.14	19.61	35.94	55.35	75.03	93.18	109.62	123.06	130.98	130.51	120.92	103.69	81.66	57.9	35.87	18.56	7.64	2.33	0.59	0.26	0.28
3.75	4.47	5.76	7.28	8.66	9.63	10.13	10.22	9.98	9.49	8.77	7.79	6.48	4.91	3.78	4.66	9.91	21.07	37.76	57.17	76.56	94.32	110.23	122.7	128.53	124.46	110.17	88.82	65.14	43.21	25.4	12.76	5.22	1.65	0.48	0.26	0.28
3.64	4.27	5.41	6.77	8.06	9.04	9.59	9.65	9.19	8.28	7.1	5.82	4.54	3.36	2.96	4.86	11.29	23.55	40.88	60.34	79.17	96.02	110.67	121.46	124.74	117.07	98.48	73.63	48.83	28.94	15.34	7.21	2.9	1	0.38	0.27	0.28
3.48	3.97	4.87	5.99	7.15	8.14	8.77	8.86	8.28	7.07	5.54	4.04	2.8	2.02	2.35	5.42	13.39	27.35	45.96	65.79	83.69	98.42	110.08	117.56	117.64	107.19	86.43	60.5	36.21	18.6	8.3	3.4	1.33	0.56	0.32	0.28	0.29
3.3	3.63	4.25	5.1	6.07	7.04	7.77	7.96	7.38	6.05	4.37	2.78	1.62	1.13	2.11	6.44	16.53	33.41	54.92	76.33	93.17	103.81	109.2	110.18	105.41	92.82	72.43	48.54	27.01	12.25	4.45	1.45	0.55	0.35	0.29	0.29	0.29
3.14	3.33	3.7	4.28	5.07	5.99	6.78	7.07	6.57	5.29	3.62	2.07	0.99	0.71	2.23	8.06	21.45	43.81	71.94	98.27	115.13	119.35	113.56	102.63	89.43	73.9	55.5	36.17	19.45	8.28	2.65	0.72	0.31	0.29	0.29	0.29	0.29
3.04	3.12	3.32	3.7	4.33	5.17	5.97	6.34	5.95	4.78	3.21	1.75	0.75	0.6	2.72	10.69	29.46	61.37	101.82	138.49	158.02	154.14	131.9	102.56	75.74	54.63	37.79	23.71	12.55	5.29	1.67	0.48	0.26	0.28	0.29	0.29	0.29
2.98	3.01	3.11	3.38	3.89	4.64	5.42	5.83	5.52	4.46	3	1.62	0.68	0.67	3.69	14.94	41.83	88.03	146.89	199.5	224.74	212.01	169.31	116.1	70.67	40.46	23.11	13.13	6.79	2.93	1.02	0.39	0.28	0.29	0.29	0.3	0.3
2.94	2.95	3.02	3.23	3.68	4.37	5.11	5.53	5.28	4.29	2.89	1.56	0.66	0.87	5.19	20.85	57.89	121.07	200.93	271.38	303.22	281.82	218.26	140.41	75.1	34.09	14.18	6.22	2.97	1.39	0.6	0.34	0.29	0.3	0.3	0.3	0.3
2.9	2.91	2.96	3.15	3.57	4.23	4.96	5.38	5.16	4.2	2.84	1.54	0.64	1.13	6.89	27.09	73.83	152.31	249.88	334.41	370.47	341.19	260.68	163.45	82.64	32.9	10.23	2.85	1.08	0.63	0.41	0.33	0.32	0.32	0.32	0.32	0.32
2.84	2.85	2.9	3.09	3.5	4.15	4.88	5.3	5.09	4.15	2.81	1.52	0.64	1.33	8.07	31.17	83.6	170.27	276.25	366.2	402.2	367.36	278.1	172.21	85.19	32.31	8.81	1.71	0.46	0.4	0.37	0.36	0.35	0.35	0.35	0.35	0.35
2.77	2.78	2.83	3.01	3.42	4.06	4.78	5.21	5.01	4.09	2.77	1.5	0.63	1.36	8.17	31.3	83.28	168.5	271.64	357.78	390.31	353.83	265.58	162.75	79.45	29.51	7.74	1.35	0.35	0.37	0.39	0.39	0.4	0.4	0.4	0.4	0.4
2.71	2.72	2.77	2.94	3.33	3.95	4.66	5.09	4.91	4.02	2.73	1.48	0.62	1.22	7.25	27.77	73.94	149.58	240.93	316.73	344.44	310.79	231.85	140.93	68.13	25	6.49	1.16	0.36	0.39	0.42	0.43	0.44	0.45	0.46	0.46	0.46
2.67	2.68	2.73	2.89	3.26	3.86	4.56	4.99	4.83	3.97	2.7	1.48	0.63	1.03	5.92	22.75	60.94	123.86	200.23	263.79	287.16	258.98	192.86	116.86	56.25	20.53	5.33	0.99	0.37	0.4	0.44	0.47	0.49	0.5	0.5	0.51	0.51
2.67	2.68	2.72	2.87	3.23	3.83	4.52	4.95	4.81	3.99	2.77	1.57	0.73	0.93	4.75	18.25	49.27	100.85	164.04	217.17	237.35	214.72	160.29	97.3	46.92	17.17	4.49	0.88	0.36	0.41	0.45	0.48	0.51	0.53	0.54	0.54	0.54
2.72	2.72	2.77	2.92	3.29	3.88	4.57	5.03	4.93	4.17	3.03	1.89	1.04	1.03	3.96	14.88	40.68	84.36	138.77	185.33	203.94	185.4	138.89	84.49	40.81	14.96	3.94	0.79	0.35	0.4	0.45	0.49	0.52	0.54	0.55	0.56	0.56
2.82	2.83	2.88	3.04	3.41	4.01	4.71	5.2	5.17	4.55	3.56	2.52	1.67	1.4	3.47	12.26	34.25	72.73	122.06	165.4	183.94	168.3	126.51	77	37.13	13.56	3.57	0.73	0.35	0.39	0.44	0.49	0.53	0.55	0.56	0.56	0.56
3	3.01	3.06	3.22	3.59	4.19	4.89	5.41	5.51	5.09	4.33	3.45	2.6	2.03	3.22	10.06	28.92	63.51	109.59	151.45	170.83	157.73	119.22	72.71	35.06	12.78	3.36	0.7	0.34	0.39	0.44	0.49	0.53	0.55	0.56	0.57	0.57
3.24	3.24	3.29	3.45	3.8	4.37	5.04	5.6	5.83	5.66	5.18	4.49	3.65	2.78	3.2	8.39	24.55	55.74	98.85	139.38	159.7	149.28	114.02	70.2	34.18	12.6	3.35	0.71	0.34	0.38	0.44	0.49	0.53	0.55	0.57	0.57	0.57
3.53	3.54	3.58	3.72	4.04	4.55	5.17	5.74	6.09	6.14	5.91	5.39	4.56	3.47	3.33	7.34	21.41	49.71	89.97	129.02	150.11	142.48	110.6	69.36	34.49	13.05	3.57	0.76	0.33	0.38	0.43	0.49	0.53	0.56	0.57	0.57	0.57
3.89	3.9	3.93	4.05	4.32	4.76	5.33	5.88	6.29	6.47	6.39	5.97	5.14	3.93	3.48	6.8	19.48	45.64	83.56	121.35	143.18	138.19	109.38	70.25	35.88	14.03	3.96	0.85	0.33	0.38	0.43	0.48	0.53	0.56	0.57	0.57	0.57
4.33	4.33	4.36	4.46	4.69	5.08	5.59	6.11	6.52	6.72	6.67	6.25	5.4	4.13	3.55	6.54	18.43	43.27	79.69	116.72	139.21	136.25	109.62	71.8	37.47	15.03	4.33	0.92	0.32	0.37	0.42	0.48	0.53	0.55	0.57	0.57	0.58
4.86	4.86	4.89	4.98	5.2	5.57	6.06	6.55	6.91	7.05	6.93	6.45	5.55	4.25	3.65	6.59	18.34	42.95	79.11	116.1	138.88	136.46	110.31	72.66	38.14	15.39	4.44	0.93	0.3	0.35	0.41	0.47	0.52	0.55	0.57	0.57	0.58
5.42	5.44	5.49	5.61	5.87	6.31	6.87	7.41	7.75	7.8	7.54	6.93	5.96	4.67	4.17	7.5	20.31	46.9	85.67	124.75	148	144.01	115.21	75.11	39.21	15.95	4.9	1.27	0.49	0.42	0.41	0.45	0.5	0.54	0.56	0.57	0.57
5.94	5.99	6.12	6.35	6.79	7.46	8.3	9.08	9.54	9.52	9.08	8.28	7.23	5.96	5.8	10.41	26.84	60.26	108.31	155.52	181.86	174.12	137.2	88.55	46.79	20.43	7.9	3.21	1.54	0.84	0.51	0.44	0.47	0.51	0.54	0.56	0.57
6.29	6.45	6.78	7.29	8.09	9.24	10.65	11.95	12.71	12.68	12	10.95	9.78	8.6	9.11	16.32	40.07	87.56	155.3	221.17	257.18	245.48	194.18	127.88	71.77	36.09	17.91	9.31	4.74	2.14	0.86	0.47	0.43	0.48	0.52	0.55	0.56
6.45	6.81	7.49	8.48	9.88	11.78	14.05	16.11	17.33	17.31	16.32	14.92	13.62	12.56	14.09	25.2	59.97	128.89	227.29	323.88	378.64	365.66	295.6	202.72	122.29	68.95	39.01	22.24	11.74	5.29	1.96	0.77	0.48	0.47	0.5	0.53	0.55
6.46	7.09	8.27	9.92	12.1	14.89	18.13	21.04	22.72	22.66	21.28	19.46	17.99	17.06	19.73	35.27	82.54	176.06	310.45	445	526.05	517.31	429.73	307.17	196.45	119.24	72.35	43.64	24.34	11.89	4.96	1.97	0.88	0.56	0.49	0.51	0.54
6.43	7.35	9.07	11.42	14.43	18.14	22.31	25.97	28	27.77	25.91	23.6	21.89	21.05	24.72	44.15	102.46	217.95	385.21	556.11	665	665.36	566.01	418.08	278.8	177.89	113.78	72.89	44.14	24.32	11.9	5.31	2.18	0.91	0.53	0.49	0.52
6.4	7.57	9.75	12.73	16.54	21.11	26.1	30.33	32.49	31.94	29.49	26.61	24.6	23.74	28.05	50.08	115.74	246.12	436.27	633.99	765.64	776.96	673.3	509.59	350.4	232.61	156.64	107.6	71.54	44.15	24.36	11.78	4.8	1.64	0.62	0.45	0.48
6.4	7.71	10.2	13.69	18.18	23.57	29.32	34	36.14	35.12	31.99	28.49	26.11	25.13	29.72	53.04	122.38	260.35	462.59	675.42	821.23	841.28	737.88	567.43	398.75	273.74	194.24	143.44	104.02	70.08	41.74	21.14	8.66	2.72	0.77	0.4	0.42
6.4	7.77	10.43	14.26	19.32	25.42	31.82	36.84	38.88	37.38	33.6	29.52	26.78	25.64	30.29	54.03	124.59	265.2	471.89	690.85	843.2	868.29	766.64	594.96	424.22	299.13	222.08	174.06	134.47	95.74	59.5	30.85	12.71	3.85	0.92	0.35	0.37
6.39	7.78	10.5	14.52	19.91	26.45	33.24	38.47	40.44	38.63	34.43	30	27.02	25.77	30.4	54.23	125.04	266.21	473.96	694.59	848.96	875.9	775.3	603.96	433.77	310.6	236.89	192.03	153.3	112.05	70.96	37.17	15.34	4.59	1.02	0.32	0.33
]'; 
fc_hc_map_cold=fliplr(fc_hc_map_cold);

% (g/s), engine out CO emissions indexed vertically by fc_map_spd and
fc_co_map_cold=1e-3*[
45.31	40.87	33.07	24.02	16.02	10.67	8.01	7.19	7.2	7.47	7.65	7.44	6.59	5.12	3.42	2.1	1.78	2.69	4.67	7.09	9.32	10.97	12.01	12.56	12.71	12.42	11.6	10.22	8.33	6.08	3.85	2.02	0.84	0.27	0.08	0.05	0.05
45.4	41.06	33.4	24.48	16.53	11.13	8.31	7.29	7.07	7.1	7.08	6.74	5.9	4.55	3.04	1.92	1.75	2.75	4.76	7.17	9.38	10.99	12.01	12.52	12.57	12.08	10.99	9.36	7.36	5.22	3.24	1.68	0.7	0.23	0.07	0.05	0.05
45.7	41.62	34.39	25.86	18.07	12.49	9.23	7.66	6.91	6.47	6.07	5.51	4.67	3.55	2.38	1.61	1.7	2.86	4.91	7.29	9.44	11	11.97	12.42	12.31	11.51	10.02	8.02	5.88	3.9	2.3	1.16	0.48	0.17	0.07	0.05	0.05
46.29	42.74	36.36	28.62	21.15	15.2	11.1	8.5	6.87	5.75	4.88	4.07	3.25	2.38	1.62	1.26	1.66	2.99	5.08	7.4	9.43	10.89	11.78	12.14	11.87	10.81	8.95	6.65	4.41	2.62	1.39	0.66	0.28	0.11	0.06	0.05	0.05
47.18	44.42	39.33	32.76	25.77	19.3	13.93	9.87	7.05	5.13	3.79	2.78	1.98	1.35	0.95	0.95	1.63	3.1	5.17	7.38	9.23	10.5	11.22	11.45	11.05	9.84	7.84	5.47	3.27	1.69	0.76	0.32	0.14	0.07	0.05	0.05	0.06
48.19	46.33	42.7	37.5	31.08	24.04	17.25	11.56	7.45	4.75	3.03	1.87	1.12	0.66	0.51	0.76	1.61	3.14	5.14	7.15	8.73	9.71	10.15	10.15	9.63	8.42	6.55	4.38	2.44	1.11	0.42	0.15	0.07	0.06	0.05	0.06	0.06
49.08	48.02	45.71	41.78	35.96	28.51	20.51	13.35	8.05	4.63	2.6	1.38	0.66	0.31	0.29	0.65	1.57	3.08	4.96	6.74	7.99	8.57	8.6	8.27	7.6	6.5	4.97	3.26	1.76	0.76	0.25	0.09	0.05	0.05	0.06	0.06	0.06
49.51	48.99	47.61	44.65	39.43	31.88	23.16	14.97	8.73	4.72	2.44	1.17	0.49	0.19	0.2	0.6	1.49	2.94	4.68	6.24	7.17	7.33	6.9	6.18	5.34	4.38	3.27	2.12	1.14	0.49	0.17	0.06	0.05	0.05	0.06	0.06	0.06
48.98	48.75	47.9	45.6	40.95	33.68	24.81	16.14	9.34	4.9	2.4	1.09	0.43	0.15	0.18	0.55	1.4	2.76	4.39	5.8	6.5	6.32	5.48	4.39	3.38	2.53	1.8	1.15	0.62	0.27	0.11	0.06	0.05	0.05	0.06	0.06	0.06
46.6	46.48	45.89	44.02	39.95	33.29	24.86	16.38	9.53	4.96	2.38	1.05	0.41	0.14	0.16	0.51	1.31	2.62	4.19	5.52	6.1	5.72	4.6	3.26	2.11	1.32	0.82	0.5	0.27	0.13	0.07	0.05	0.05	0.05	0.06	0.06	0.06
41.62	41.55	41.09	39.56	36.12	30.37	22.95	15.35	9.07	4.8	2.33	1.03	0.4	0.14	0.16	0.49	1.28	2.57	4.14	5.47	6.01	5.54	4.27	2.77	1.52	0.73	0.34	0.17	0.1	0.06	0.05	0.05	0.05	0.05	0.06	0.06	0.06
34.05	33.99	33.63	32.43	29.72	25.18	19.28	13.17	8.05	4.45	2.26	1.04	0.41	0.14	0.16	0.5	1.32	2.65	4.28	5.67	6.23	5.71	4.35	2.73	1.39	0.56	0.19	0.07	0.04	0.04	0.04	0.05	0.05	0.05	0.06	0.06	0.06
25.23	25.18	24.93	24.08	22.18	19	14.89	10.59	6.88	4.11	2.25	1.09	0.44	0.15	0.17	0.54	1.42	2.86	4.62	6.13	6.75	6.19	4.72	2.95	1.48	0.57	0.17	0.05	0.03	0.04	0.04	0.05	0.05	0.05	0.06	0.06	0.06
17.03	17	16.85	16.35	15.25	13.41	11.02	8.43	6.02	3.97	2.36	1.2	0.49	0.17	0.19	0.6	1.58	3.19	5.15	6.83	7.53	6.92	5.29	3.31	1.66	0.65	0.19	0.05	0.03	0.03	0.04	0.05	0.05	0.05	0.06	0.06	0.06
10.94	10.93	10.86	10.66	10.21	9.45	8.41	7.13	5.67	4.11	2.61	1.39	0.6	0.22	0.23	0.69	1.77	3.57	5.77	7.64	8.41	7.72	5.89	3.68	1.84	0.71	0.2	0.05	0.03	0.04	0.04	0.05	0.05	0.05	0.05	0.06	0.06
7.21	7.21	7.21	7.21	7.21	7.19	7.06	6.64	5.8	4.51	3.05	1.74	0.85	0.39	0.34	0.79	1.94	3.88	6.27	8.3	9.13	8.35	6.34	3.94	1.96	0.75	0.21	0.05	0.03	0.04	0.04	0.05	0.05	0.05	0.05	0.05	0.05
5.35	5.36	5.39	5.51	5.78	6.19	6.6	6.72	6.27	5.2	3.79	2.44	1.44	0.84	0.63	0.93	2.01	3.97	6.45	8.6	9.47	8.65	6.53	4.02	1.97	0.74	0.21	0.05	0.03	0.04	0.04	0.05	0.05	0.05	0.05	0.05	0.05
4.64	4.65	4.71	4.89	5.3	5.95	6.68	7.14	7.01	6.19	4.95	3.64	2.54	1.7	1.18	1.16	1.99	3.82	6.28	8.49	9.43	8.64	6.51	3.98	1.93	0.72	0.2	0.05	0.03	0.04	0.04	0.05	0.05	0.05	0.05	0.05	0.05
4.61	4.62	4.69	4.91	5.39	6.17	7.08	7.77	7.93	7.45	6.51	5.35	4.17	3	2.01	1.52	1.96	3.53	5.9	8.12	9.15	8.45	6.39	3.9	1.89	0.7	0.19	0.05	0.04	0.04	0.04	0.05	0.05	0.05	0.05	0.05	0.05
5.01	5.02	5.09	5.32	5.83	6.65	7.64	8.49	8.93	8.82	8.26	7.33	6.07	4.54	3.01	1.97	1.99	3.28	5.52	7.74	8.87	8.3	6.35	3.92	1.92	0.72	0.2	0.05	0.04	0.04	0.04	0.04	0.05	0.05	0.05	0.05	0.05
5.71	5.72	5.79	6.01	6.5	7.31	8.33	9.29	9.97	10.23	10.03	9.31	7.97	6.07	4.02	2.47	2.1	3.16	5.33	7.59	8.85	8.43	6.57	4.14	2.08	0.8	0.23	0.06	0.04	0.04	0.04	0.04	0.05	0.05	0.05	0.05	0.05
6.72	6.73	6.79	7	7.47	8.26	9.29	10.35	11.21	11.72	11.76	11.14	9.68	7.43	4.92	2.95	2.31	3.27	5.5	7.93	9.38	9.1	7.24	4.68	2.41	0.96	0.28	0.07	0.03	0.04	0.04	0.04	0.04	0.05	0.05	0.05	0.05
8.15	8.16	8.21	8.42	8.89	9.7	10.78	11.91	12.87	13.47	13.57	12.89	11.23	8.63	5.72	3.4	2.6	3.63	6.11	8.87	10.6	10.4	8.4	5.52	2.9	1.17	0.34	0.08	0.03	0.04	0.04	0.04	0.04	0.04	0.04	0.05	0.05
10.14	10.16	10.23	10.45	10.98	11.88	13.08	14.31	15.3	15.83	15.77	14.85	12.9	9.97	6.72	4.17	3.35	4.62	7.54	10.77	12.76	12.47	10.04	6.59	3.45	1.39	0.4	0.09	0.03	0.03	0.04	0.04	0.04	0.04	0.04	0.04	0.04
12.67	12.72	12.87	13.21	13.92	15.09	16.58	18.05	19.1	19.45	19.05	17.76	15.49	12.31	8.92	6.32	5.67	7.38	10.93	14.68	16.76	15.96	12.6	8.15	4.25	1.75	0.57	0.17	0.07	0.05	0.04	0.04	0.04	0.04	0.04	0.04	0.04
15.38	15.58	16	16.73	17.96	19.78	22.04	24.17	25.54	25.76	24.91	23.13	20.6	17.43	14.24	11.97	11.83	14.28	18.72	23.07	24.96	22.96	17.76	11.45	6.19	2.89	1.28	0.6	0.3	0.14	0.07	0.04	0.04	0.04	0.04	0.04	0.04
17.65	18.23	19.34	20.97	23.31	26.45	30.2	33.66	35.81	36.05	34.68	32.34	29.67	26.86	24.36	22.92	23.72	27.35	33.09	38.29	39.87	36.06	28	18.68	11.01	6.08	3.38	1.9	0.98	0.42	0.14	0.05	0.04	0.04	0.04	0.04	0.04
19.03	20.28	22.62	25.84	30.01	35.21	41.18	46.62	50	50.4	48.45	45.5	42.79	40.68	39.34	39.18	41.37	46.6	54.13	60.66	62.21	56.42	44.79	31.37	20.09	12.43	7.69	4.62	2.47	1.09	0.38	0.12	0.05	0.04	0.04	0.04	0.04
19.47	21.65	25.67	31	37.52	45.17	53.62	61.18	65.83	66.36	63.79	60.2	57.52	56.25	56.25	57.57	61.29	68.31	77.91	86.29	88.48	81.35	66.46	48.75	33.23	22.03	14.46	9.1	5.14	2.5	1.02	0.38	0.15	0.07	0.04	0.04	0.04
19.35	22.49	28.24	35.74	44.6	54.57	65.2	74.47	80.05	80.54	77.33	73.15	70.49	69.97	71.17	73.79	78.85	87.46	99.02	109.41	112.87	105.43	88.36	67.19	47.84	33.24	22.88	15.24	9.33	5.13	2.5	1.1	0.42	0.15	0.06	0.04	0.04
19.09	22.95	30.04	39.27	50.01	61.77	73.91	84.2	90.13	90.33	86.51	81.83	79.16	79.13	81.11	84.6	90.56	100.25	113.25	125.37	130.33	123.5	105.63	82.49	60.64	43.8	31.65	22.54	15.14	9.34	5.14	2.47	0.98	0.31	0.08	0.04	0.04
18.85	23.08	30.88	41.14	53.13	66.14	79.27	90.06	95.92	95.69	91.32	86.27	83.52	83.71	86.09	90.01	96.41	106.67	120.49	133.73	139.87	133.88	116.07	92.24	69.4	51.84	39.41	30.1	22.03	14.83	8.82	4.45	1.81	0.54	0.12	0.04	0.04
18.67	22.96	30.97	41.7	54.4	68.26	82.02	93.02	98.65	97.97	93.16	87.84	85.01	85.25	87.75	91.81	98.36	108.83	122.98	136.76	143.58	138.23	120.74	96.95	74.1	56.9	45.21	36.57	28.48	20.27	12.59	6.51	2.67	0.78	0.16	0.03	0.03
18.56	22.81	30.8	41.66	54.7	69.01	83.13	94.23	99.71	98.75	93.69	88.22	85.32	85.56	88.08	92.17	98.75	109.26	123.51	137.47	144.54	139.45	122.16	98.51	75.91	59.23	48.31	40.37	32.47	23.73	15.02	7.85	3.23	0.94	0.18	0.03	0.03
]'; 
fc_co_map_cold=fliplr(fc_co_map_cold);

% (g/s), engine out NOx emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_nox_map_cold=1e-3*[
61	66.72	76.8	88.49	98.83	105.72	109.09	109.94	109.53	108.11	104.81	97.32	83.66	63.86	42.1	24.66	17.72	23.16	38.76	58.4	76.17	88.27	94.62	96.71	95.95	92.48	85.83	75.43	61.38	44.83	28.38	14.86	6.14	1.86	0.45	0.19	0.21
60.87	66.48	76.38	87.9	98.16	105.06	108.35	108.64	106.86	103.31	97.74	88.75	75.1	56.81	37.42	22.35	17.15	23.59	39.54	59.12	76.65	88.51	94.66	96.43	94.89	89.96	81.29	69.08	54.28	38.49	23.84	12.34	5.09	1.56	0.41	0.2	0.21
60.49	65.76	75.1	86.1	96.13	103.15	106.49	106.01	102	94.79	85.25	73.65	60.01	44.41	29.21	18.34	16.27	24.5	41.07	60.54	77.57	88.9	94.58	95.77	92.99	85.82	74.1	59.18	43.3	28.72	16.89	8.49	3.48	1.11	0.34	0.2	0.21
59.72	64.31	72.53	82.5	92.07	99.35	103.14	102.23	95.98	84.74	70.7	56.12	42.52	30.05	19.75	13.86	15.56	26.03	43.35	62.56	78.71	89.06	93.83	94.1	89.95	80.66	66.26	49.08	32.46	19.24	10.2	4.8	1.94	0.68	0.27	0.21	0.22
58.57	62.13	68.68	77.07	85.91	93.61	98.35	97.7	89.99	75.51	57.67	40.55	27.04	17.39	11.48	10.1	15.31	27.91	45.89	64.58	79.37	88.03	91.1	89.85	84.31	73.71	58.13	40.33	24.07	12.37	5.52	2.27	0.9	0.39	0.24	0.21	0.22
57.27	59.66	64.29	70.83	78.75	86.83	92.74	92.98	84.9	68.7	48.61	29.99	16.65	8.94	6.04	7.83	15.59	29.75	48.05	65.84	78.65	84.71	84.99	81.38	74.37	63.45	48.64	32.35	17.96	8.14	2.97	0.98	0.39	0.26	0.22	0.22	0.22
56.12	57.48	60.37	65.16	72.04	80.21	87.03	88.27	80.63	64.14	43.39	24.36	11.31	4.69	3.35	6.77	15.82	30.66	48.73	65.24	75.63	78.37	75.01	68.34	59.79	49.41	37.01	24.07	12.93	5.51	1.77	0.5	0.22	0.22	0.22	0.22	0.22
55.33	55.98	57.65	61.07	66.91	74.68	81.73	83.61	76.69	60.8	40.52	21.9	9.3	3.19	2.4	6.27	15.49	30.1	47.39	62.38	70.35	69.78	62.75	53	43.11	33.75	24.49	15.68	8.34	3.52	1.12	0.34	0.2	0.21	0.22	0.22	0.23
54.85	55.13	56.08	58.57	63.42	70.37	77	78.98	72.65	57.69	38.39	20.6	8.56	2.78	2.06	5.71	14.36	28.04	44.15	57.69	63.79	60.8	51.04	38.99	28.2	19.93	13.57	8.47	4.51	1.95	0.69	0.28	0.21	0.22	0.22	0.23	0.23
54.42	54.56	55.18	57.09	61.1	67.09	72.94	74.62	68.66	54.58	36.38	19.54	8.13	2.62	1.85	5.08	12.9	25.42	40.27	52.62	57.6	53.38	42.3	29.14	18.05	10.65	6.29	3.67	1.97	0.92	0.41	0.24	0.22	0.23	0.23	0.23	0.23
53.75	53.84	54.31	55.86	59.22	64.32	69.31	70.52	64.72	51.41	34.27	18.41	7.66	2.45	1.65	4.5	11.57	23	36.7	48.11	52.51	47.98	36.74	23.54	12.67	5.91	2.63	1.28	0.71	0.42	0.28	0.24	0.24	0.24	0.24	0.24	0.24
52.59	52.66	53.06	54.33	57.13	61.35	65.37	65.92	60.15	47.63	31.69	17.01	7.08	2.25	1.49	4.07	10.5	20.96	33.54	44.01	47.95	43.53	32.85	20.37	10.21	4.07	1.34	0.49	0.3	0.27	0.26	0.26	0.26	0.27	0.27	0.27	0.27
50.78	50.83	51.13	52.1	54.19	57.28	59.99	59.63	53.84	42.35	28.08	15.04	6.25	1.98	1.31	3.6	9.34	18.68	29.92	39.23	42.65	38.56	28.88	17.68	8.66	3.28	0.97	0.31	0.23	0.25	0.27	0.29	0.3	0.31	0.31	0.31	0.31
48.28	48.31	48.49	49.04	50.2	51.78	52.74	51.15	45.35	35.25	23.21	12.39	5.14	1.63	1.09	3.02	7.88	15.81	25.36	33.26	36.1	32.53	24.25	14.74	7.16	2.7	0.81	0.29	0.23	0.26	0.29	0.32	0.34	0.35	0.36	0.36	0.36
45.23	45.24	45.28	45.37	45.49	45.4	44.46	41.56	35.81	27.33	17.82	9.49	3.95	1.28	0.87	2.4	6.28	12.65	20.36	26.75	29.07	26.18	19.48	11.81	5.72	2.15	0.66	0.26	0.23	0.27	0.31	0.34	0.37	0.39	0.4	0.4	0.4
42.03	42.02	41.94	41.62	40.85	39.34	36.78	32.81	27.22	20.29	13.13	7.09	3.1	1.14	0.81	1.93	4.9	9.85	15.89	20.94	22.81	20.59	15.35	9.33	4.53	1.72	0.54	0.22	0.21	0.26	0.31	0.35	0.39	0.41	0.42	0.42	0.42
39.01	38.98	38.82	38.24	36.88	34.45	30.92	26.4	21.16	15.57	10.28	5.97	3.08	1.52	1.07	1.74	3.91	7.75	12.56	16.63	18.2	16.49	12.34	7.52	3.66	1.39	0.44	0.19	0.19	0.24	0.29	0.34	0.38	0.41	0.42	0.42	0.42
36.33	36.3	36.08	35.37	33.73	30.89	26.99	22.45	17.79	13.37	9.51	6.4	4.15	2.63	1.8	1.87	3.31	6.32	10.32	13.82	15.26	13.9	10.43	6.36	3.09	1.17	0.37	0.16	0.16	0.21	0.26	0.31	0.36	0.38	0.4	0.4	0.4
33.81	33.77	33.54	32.78	31.06	28.18	24.36	20.2	16.32	13.02	10.35	8.12	6.19	4.42	2.97	2.28	3.01	5.38	8.91	12.17	13.65	12.57	9.49	5.8	2.82	1.07	0.33	0.14	0.14	0.18	0.23	0.28	0.32	0.35	0.36	0.36	0.36
31.2	31.16	30.93	30.19	28.54	25.82	22.34	18.74	15.71	13.47	11.83	10.33	8.57	6.43	4.29	2.85	2.93	4.84	8.12	11.35	12.98	12.13	9.28	5.74	2.82	1.08	0.33	0.13	0.12	0.16	0.2	0.24	0.28	0.31	0.32	0.32	0.32
28.32	28.29	28.08	27.42	25.95	23.53	20.5	17.52	15.26	13.91	13.11	12.17	10.54	8.09	5.39	3.36	2.98	4.6	7.77	11.06	12.88	12.26	9.55	6.03	3.03	1.18	0.36	0.13	0.11	0.14	0.18	0.22	0.25	0.27	0.28	0.28	0.28
25.15	25.12	24.96	24.41	23.19	21.18	18.65	16.25	14.6	13.85	13.6	13.04	11.52	8.94	5.97	3.63	3.01	4.47	7.58	10.93	12.91	12.49	9.92	6.41	3.3	1.32	0.41	0.13	0.1	0.13	0.16	0.2	0.23	0.25	0.26	0.26	0.26
21.77	21.75	21.63	21.22	20.26	18.67	16.67	14.79	13.6	13.23	13.26	12.91	11.51	8.98	6	3.63	2.93	4.3	7.31	10.61	12.66	12.41	10.01	6.57	3.45	1.4	0.43	0.13	0.09	0.12	0.15	0.18	0.21	0.23	0.24	0.25	0.25
18.26	18.24	18.16	17.86	17.16	15.98	14.47	13.09	12.27	12.16	12.36	12.13	10.87	8.5	5.69	3.44	2.76	4.02	6.83	9.94	11.91	11.74	9.52	6.31	3.34	1.37	0.42	0.12	0.08	0.11	0.14	0.17	0.21	0.23	0.24	0.24	0.24
14.76	14.75	14.68	14.47	13.97	13.14	12.1	11.19	10.75	10.88	11.21	11.1	10.01	7.87	5.3	3.24	2.6	3.71	6.22	8.99	10.73	10.53	8.52	5.63	2.99	1.24	0.4	0.13	0.08	0.1	0.13	0.16	0.19	0.22	0.23	0.24	0.24
11.48	11.45	11.36	11.18	10.84	10.3	9.67	9.19	9.1	9.46	9.93	9.95	9.05	7.2	4.95	3.13	2.56	3.51	5.65	7.98	9.37	9.09	7.3	4.83	2.64	1.2	0.49	0.22	0.13	0.11	0.12	0.15	0.18	0.2	0.22	0.23	0.24
8.75	8.67	8.52	8.3	8	7.65	7.33	7.18	7.37	7.88	8.44	8.58	7.93	6.46	4.64	3.17	2.72	3.5	5.24	7.09	8.12	7.8	6.32	4.38	2.7	1.57	0.92	0.54	0.31	0.17	0.12	0.13	0.15	0.18	0.21	0.22	0.23
6.83	6.66	6.37	6.02	5.67	5.38	5.22	5.26	5.57	6.11	6.68	6.92	6.56	5.58	4.34	3.34	3.08	3.71	5.03	6.39	7.14	6.91	5.91	4.64	3.52	2.64	1.9	1.25	0.7	0.34	0.17	0.12	0.13	0.16	0.19	0.21	0.23
5.79	5.52	5.03	4.46	3.95	3.61	3.47	3.55	3.86	4.34	4.85	5.16	5.1	4.66	4.04	3.56	3.52	4.03	4.96	5.92	6.51	6.54	6.2	5.7	5.17	4.47	3.52	2.42	1.42	0.71	0.33	0.17	0.14	0.15	0.18	0.2	0.22
5.4	5.01	4.32	3.52	2.81	2.35	2.16	2.2	2.45	2.84	3.27	3.63	3.83	3.85	3.79	3.76	3.92	4.33	4.96	5.63	6.19	6.6	6.96	7.25	7.27	6.74	5.58	4.04	2.55	1.42	0.72	0.35	0.19	0.15	0.16	0.19	0.2
5.35	4.86	3.99	3	2.12	1.54	1.29	1.3	1.5	1.82	2.21	2.61	2.98	3.31	3.62	3.9	4.19	4.54	4.98	5.5	6.11	6.88	7.85	8.79	9.3	8.98	7.77	5.98	4.12	2.55	1.42	0.71	0.33	0.18	0.15	0.17	0.18
5.37	4.81	3.84	2.72	1.74	1.09	0.8	0.8	0.98	1.29	1.67	2.09	2.55	3.04	3.53	3.96	4.33	4.65	5	5.46	6.14	7.17	8.52	9.9	10.8	10.77	9.75	8.01	5.98	4.04	2.42	1.25	0.54	0.23	0.14	0.14	0.15
5.38	4.79	3.77	2.59	1.55	0.88	0.58	0.58	0.77	1.08	1.46	1.91	2.4	2.95	3.5	3.99	4.37	4.69	5.01	5.46	6.2	7.35	8.9	10.51	11.68	11.96	11.26	9.74	7.72	5.51	3.43	1.8	0.77	0.28	0.12	0.11	0.11
5.37	4.78	3.74	2.54	1.49	0.8	0.5	0.51	0.7	1.02	1.41	1.86	2.37	2.93	3.5	3.99	4.38	4.69	5.02	5.47	6.22	7.42	9.03	10.74	12.05	12.53	12.08	10.77	8.8	6.44	4.09	2.16	0.91	0.31	0.12	0.09	0.09
]';  
fc_nox_map_cold=fliplr(fc_nox_map_cold);

% (g/s), engine out PM emissions indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_pm_map_cold=1e-3*[
1.29	1.27	1.27	1.31	1.44	1.66	1.96	2.3	2.67	3.09	3.58	4.23	5.09	6.18	7.32	8.31	8.96	9.28	9.4	9.49	9.74	10.24	11.01	11.97	13.01	14.15	15.5	17.27	19.55	22.25	25.03	27.59	29.74	31.52	32.97	34.06	34.65
1.43	1.4	1.36	1.35	1.41	1.57	1.79	2.08	2.42	2.83	3.33	3.99	4.87	5.99	7.17	8.18	8.85	9.19	9.33	9.44	9.71	10.22	11.02	12.02	13.19	14.55	16.23	18.29	20.72	23.33	25.89	28.18	30.11	31.74	33.1	34.13	34.69
1.68	1.62	1.52	1.43	1.39	1.43	1.56	1.76	2.04	2.42	2.91	3.57	4.47	5.61	6.81	7.86	8.55	8.93	9.1	9.27	9.58	10.15	10.99	12.07	13.44	15.17	17.35	19.88	22.53	25.05	27.27	29.17	30.79	32.2	33.39	34.29	34.77
1.97	1.87	1.71	1.53	1.39	1.33	1.36	1.48	1.7	2.03	2.47	3.09	3.96	5.07	6.25	7.27	7.98	8.37	8.6	8.84	9.25	9.89	10.82	12.03	13.63	15.75	18.44	21.42	24.32	26.78	28.75	30.34	31.68	32.84	33.8	34.52	34.9
2.23	2.11	1.89	1.64	1.43	1.29	1.26	1.32	1.48	1.74	2.11	2.66	3.43	4.43	5.5	6.44	7.1	7.5	7.77	8.09	8.59	9.35	10.4	11.76	13.6	16.06	19.15	22.49	25.58	28.05	29.93	31.38	32.57	33.53	34.27	34.78	35.03
2.41	2.28	2.04	1.76	1.51	1.33	1.24	1.25	1.36	1.57	1.87	2.32	2.97	3.8	4.69	5.47	6.03	6.4	6.69	7.05	7.63	8.49	9.66	11.19	13.25	15.97	19.33	22.88	26.11	28.67	30.61	32.1	33.27	34.12	34.69	35.01	35.16
2.52	2.39	2.17	1.89	1.63	1.41	1.27	1.23	1.3	1.47	1.73	2.1	2.6	3.25	3.93	4.53	4.96	5.26	5.53	5.9	6.51	7.43	8.7	10.36	12.55	15.42	18.88	22.51	25.83	28.53	30.67	32.36	33.64	34.48	34.95	35.16	35.24
2.58	2.48	2.29	2.05	1.78	1.52	1.32	1.22	1.24	1.39	1.63	1.95	2.35	2.82	3.31	3.74	4.05	4.27	4.49	4.82	5.42	6.36	7.67	9.37	11.58	14.41	17.78	21.32	24.65	27.54	30.02	32.07	33.63	34.59	35.07	35.24	35.28
2.62	2.55	2.41	2.21	1.94	1.64	1.36	1.19	1.17	1.32	1.56	1.87	2.19	2.54	2.88	3.18	3.38	3.53	3.68	3.97	4.53	5.44	6.72	8.36	10.44	13.03	16.07	19.33	22.55	25.64	28.57	31.16	33.15	34.38	34.96	35.15	35.18
2.65	2.61	2.52	2.35	2.08	1.73	1.39	1.15	1.11	1.27	1.55	1.86	2.16	2.43	2.66	2.86	2.98	3.06	3.15	3.38	3.87	4.72	5.92	7.42	9.24	11.43	13.95	16.71	19.67	22.87	26.25	29.45	32	33.6	34.35	34.59	34.63
2.67	2.65	2.58	2.44	2.17	1.8	1.41	1.13	1.09	1.28	1.61	1.97	2.27	2.5	2.68	2.81	2.88	2.9	2.93	3.08	3.48	4.22	5.28	6.58	8.07	9.77	11.66	13.78	16.31	19.42	23.07	26.75	29.77	31.7	32.62	32.92	32.97
2.65	2.64	2.59	2.47	2.22	1.85	1.45	1.17	1.14	1.38	1.77	2.18	2.51	2.76	2.92	3.03	3.07	3.06	3.03	3.1	3.4	4	4.86	5.9	7.02	8.18	9.42	10.87	12.84	15.63	19.21	22.99	26.19	28.26	29.27	29.59	29.66
2.56	2.56	2.52	2.42	2.21	1.87	1.51	1.25	1.25	1.52	1.96	2.43	2.82	3.11	3.32	3.46	3.51	3.49	3.43	3.44	3.64	4.08	4.72	5.45	6.17	6.84	7.5	8.34	9.74	12	15.12	18.52	21.45	23.37	24.31	24.62	24.69
2.39	2.38	2.36	2.28	2.11	1.85	1.54	1.34	1.37	1.66	2.12	2.62	3.05	3.41	3.7	3.91	4.02	4.03	3.99	3.99	4.12	4.41	4.82	5.26	5.63	5.9	6.14	6.54	7.43	9.1	11.5	14.17	16.48	18.01	18.75	19	19.05
2.12	2.11	2.1	2.05	1.93	1.74	1.53	1.4	1.46	1.74	2.18	2.67	3.13	3.55	3.91	4.2	4.39	4.48	4.51	4.55	4.65	4.83	5.05	5.26	5.4	5.45	5.48	5.66	6.22	7.34	8.99	10.8	12.37	13.39	13.89	14.05	14.08
1.8	1.8	1.79	1.76	1.7	1.59	1.47	1.42	1.51	1.77	2.15	2.59	3.02	3.45	3.85	4.19	4.44	4.62	4.76	4.9	5.04	5.18	5.29	5.37	5.41	5.43	5.47	5.63	6.03	6.76	7.79	8.89	9.81	10.4	10.68	10.77	10.79
1.49	1.49	1.49	1.48	1.46	1.43	1.4	1.42	1.53	1.76	2.06	2.41	2.78	3.16	3.54	3.87	4.16	4.42	4.69	4.95	5.18	5.34	5.43	5.5	5.58	5.7	5.89	6.15	6.52	7.03	7.64	8.24	8.72	9.02	9.16	9.2	9.21
1.22	1.22	1.22	1.24	1.26	1.29	1.34	1.43	1.56	1.73	1.95	2.19	2.47	2.77	3.07	3.37	3.65	3.97	4.34	4.73	5.07	5.31	5.46	5.59	5.78	6.07	6.45	6.87	7.29	7.68	8.05	8.36	8.59	8.73	8.79	8.81	8.81
1	1	1.01	1.04	1.09	1.19	1.31	1.44	1.57	1.7	1.82	1.96	2.14	2.35	2.58	2.82	3.09	3.43	3.86	4.34	4.78	5.11	5.37	5.61	5.94	6.4	6.96	7.53	8.01	8.35	8.59	8.74	8.84	8.9	8.92	8.93	8.93
0.83	0.83	0.84	0.88	0.96	1.1	1.27	1.45	1.57	1.65	1.69	1.74	1.83	1.98	2.15	2.35	2.59	2.92	3.36	3.88	4.38	4.8	5.17	5.54	6	6.6	7.29	7.95	8.48	8.82	9	9.09	9.14	9.17	9.18	9.19	9.19
0.7	0.7	0.72	0.76	0.86	1.03	1.23	1.43	1.55	1.58	1.55	1.53	1.57	1.67	1.81	1.97	2.18	2.49	2.91	3.41	3.93	4.42	4.89	5.38	5.95	6.63	7.37	8.08	8.62	8.98	9.17	9.28	9.35	9.39	9.41	9.42	9.42
0.63	0.63	0.65	0.69	0.79	0.95	1.16	1.36	1.46	1.47	1.41	1.35	1.35	1.41	1.52	1.66	1.85	2.12	2.51	2.99	3.5	4.03	4.56	5.14	5.78	6.49	7.24	7.92	8.46	8.84	9.09	9.28	9.41	9.51	9.55	9.57	9.57
0.59	0.59	0.6	0.64	0.72	0.86	1.05	1.21	1.31	1.3	1.24	1.17	1.14	1.18	1.26	1.36	1.53	1.78	2.15	2.61	3.12	3.66	4.24	4.86	5.54	6.26	6.97	7.62	8.16	8.57	8.9	9.18	9.4	9.54	9.62	9.64	9.65
0.55	0.55	0.56	0.58	0.64	0.74	0.88	1.01	1.08	1.09	1.04	0.99	0.96	0.96	0.99	1.06	1.19	1.44	1.8	2.26	2.78	3.34	3.95	4.6	5.29	6	6.72	7.38	7.93	8.39	8.77	9.1	9.37	9.55	9.64	9.67	9.67
0.48	0.48	0.49	0.5	0.54	0.61	0.7	0.79	0.86	0.88	0.87	0.84	0.8	0.77	0.76	0.78	0.89	1.12	1.49	1.97	2.51	3.09	3.71	4.37	5.06	5.8	6.56	7.29	7.92	8.42	8.83	9.15	9.4	9.57	9.65	9.67	9.68
0.39	0.39	0.4	0.41	0.45	0.5	0.58	0.67	0.74	0.78	0.79	0.78	0.74	0.68	0.62	0.6	0.67	0.88	1.25	1.72	2.28	2.89	3.52	4.18	4.87	5.64	6.5	7.35	8.09	8.64	9.03	9.29	9.48	9.6	9.66	9.68	9.68
0.28	0.29	0.32	0.36	0.41	0.49	0.6	0.71	0.8	0.86	0.87	0.84	0.79	0.7	0.62	0.57	0.61	0.79	1.1	1.54	2.08	2.69	3.32	3.96	4.66	5.48	6.44	7.43	8.29	8.9	9.27	9.46	9.58	9.64	9.67	9.68	9.68
0.2	0.23	0.29	0.37	0.48	0.62	0.78	0.95	1.07	1.12	1.1	1.03	0.95	0.85	0.75	0.69	0.71	0.82	1.05	1.4	1.87	2.42	3.03	3.65	4.35	5.22	6.28	7.41	8.39	9.06	9.43	9.58	9.65	9.67	9.68	9.68	9.68
0.15	0.21	0.32	0.46	0.64	0.85	1.1	1.32	1.46	1.48	1.41	1.29	1.16	1.05	0.96	0.89	0.88	0.93	1.06	1.28	1.63	2.08	2.62	3.21	3.91	4.82	5.97	7.21	8.3	9.05	9.45	9.62	9.67	9.69	9.69	9.68	9.68
0.14	0.23	0.39	0.61	0.86	1.16	1.48	1.75	1.89	1.87	1.73	1.54	1.37	1.24	1.15	1.09	1.05	1.04	1.07	1.17	1.38	1.72	2.17	2.71	3.39	4.31	5.49	6.79	7.96	8.8	9.3	9.54	9.65	9.68	9.69	9.69	9.68
0.15	0.26	0.48	0.76	1.1	1.48	1.87	2.18	2.32	2.24	2.01	1.74	1.52	1.38	1.29	1.23	1.17	1.12	1.08	1.08	1.18	1.41	1.78	2.25	2.88	3.74	4.86	6.12	7.33	8.29	8.95	9.36	9.57	9.66	9.69	9.69	9.69
0.16	0.3	0.55	0.89	1.31	1.77	2.23	2.58	2.7	2.56	2.24	1.89	1.62	1.45	1.35	1.29	1.23	1.16	1.08	1.03	1.06	1.22	1.52	1.93	2.48	3.21	4.17	5.29	6.47	7.55	8.44	9.07	9.45	9.63	9.69	9.7	9.69
0.17	0.32	0.6	0.99	1.46	2	2.53	2.92	3.03	2.83	2.42	1.99	1.67	1.48	1.38	1.32	1.25	1.18	1.08	1	1	1.13	1.39	1.75	2.21	2.79	3.54	4.49	5.61	6.8	7.91	8.78	9.33	9.6	9.69	9.7	9.7
0.17	0.33	0.62	1.03	1.55	2.14	2.71	3.12	3.22	2.98	2.53	2.05	1.7	1.49	1.38	1.32	1.26	1.18	1.08	1	0.99	1.1	1.34	1.68	2.09	2.56	3.17	3.99	5.06	6.32	7.56	8.59	9.25	9.58	9.68	9.7	9.7
]';
fc_pm_map_cold=fliplr(fc_pm_map_cold);

% (C), engine exhaust gas temperature indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_extmp_map_cold=[
455.58	455.57	455.52	455.32	454.73	453.33	450.57	445.96	439.34	431.11	422.25	414.01	407.52	403.49	402.16	403.34	406.6	411.25	416.5	421.59	426.04	429.88	433.68	438.35	444.7	453	462.78	472.94	482.2	489.66	495.15	499.21	502.83	506.95	512.09	518.14	524.52	530.45	535.36	539.13	542.05	544.66	547.37	550.33	553.4	556.28	558.78	560.97	563.21	565.97	569.64	574.29	579.59	584.95	589.82	593.99	597.73	601.68	606.55	612.65	619.69	626.73	632.47	635.71
455.58	455.57	455.52	455.32	454.75	453.38	450.67	446.11	439.54	431.34	422.46	414.14	407.51	403.31	401.75	402.67	405.61	409.92	414.83	419.63	423.87	427.6	431.38	436.11	442.53	450.91	460.73	470.88	480.09	487.49	492.93	497.01	500.71	504.97	510.27	516.51	523.04	529.1	534.1	537.93	540.92	543.6	546.41	549.48	552.65	555.63	558.23	560.52	562.87	565.77	569.6	574.38	579.77	585.17	590.04	594.19	597.94	601.94	606.88	613.05	620.12	627.15	632.86	636.07
455.58	455.57	455.52	455.34	454.79	453.48	450.86	446.42	439.95	431.8	422.86	414.36	407.45	402.88	400.89	401.3	403.65	407.33	411.62	415.88	419.72	423.23	426.96	431.73	438.25	446.69	456.49	466.54	475.58	482.81	488.16	492.25	496.1	500.62	506.27	512.85	519.7	525.99	531.16	535.13	538.25	541.1	544.11	547.41	550.82	554.02	556.82	559.32	561.91	565.09	569.22	574.29	579.89	585.4	590.3	594.46	598.25	602.36	607.45	613.76	620.9	627.93	633.58	636.74
455.57	455.56	455.52	455.36	454.85	453.63	451.14	446.87	440.55	432.44	423.38	414.58	407.22	402.09	399.46	399.16	400.74	403.59	407.06	410.61	413.92	417.11	420.69	425.43	431.93	440.28	449.88	459.61	468.29	475.19	480.35	484.43	488.48	493.38	499.5	506.58	513.85	520.47	525.88	530.03	533.36	536.48	539.83	543.51	547.31	550.86	553.99	556.81	559.77	563.39	568	573.54	579.5	585.25	590.27	594.52	598.43	602.74	608.07	614.6	621.88	628.92	634.51	637.61
455.54	455.54	455.5	455.35	454.91	453.79	451.47	447.4	441.25	433.17	423.92	414.68	406.68	400.79	397.35	396.2	396.89	398.83	401.41	404.16	406.86	409.63	412.96	417.51	423.78	431.77	440.85	449.95	457.99	464.38	469.22	473.26	477.53	482.85	489.52	497.16	504.93	511.91	517.59	521.98	525.6	529.09	532.93	537.16	541.51	545.57	549.14	552.41	555.86	560.05	565.31	571.48	577.98	584.12	589.39	593.84	597.98	602.61	608.32	615.21	622.73	629.87	635.45	638.51
455.43	455.42	455.39	455.27	454.88	453.88	451.75	447.9	441.9	433.81	424.28	414.46	405.65	398.84	394.46	392.4	392.19	393.23	394.95	396.94	399.02	401.32	404.28	408.44	414.21	421.5	429.7	437.82	444.94	450.59	454.99	458.92	463.38	469.12	476.32	484.49	492.7	500.03	505.96	510.61	514.57	518.55	523.01	527.94	532.98	537.67	541.79	545.58	549.6	554.45	560.44	567.34	574.49	581.11	586.75	591.5	596.01	601.12	607.41	614.89	622.89	630.32	636.04	639.14
455.09	455.09	455.06	454.96	454.63	453.75	451.79	448.14	442.28	434.13	424.23	413.74	404	396.16	390.79	387.82	386.8	387.09	388.11	389.47	391.01	392.85	395.35	398.96	403.95	410.22	417.19	424.03	429.98	434.73	438.57	442.3	446.84	452.88	460.47	469.03	477.56	485.11	491.24	496.12	500.44	504.96	510.11	515.82	521.63	527	531.72	536.04	540.59	546.04	552.66	560.19	567.87	574.9	580.87	585.98	591	596.83	604.04	612.51	621.43	629.59	635.77	639.08
454.27	454.26	454.25	454.17	453.88	453.1	451.29	447.81	442.05	433.8	423.53	412.33	401.67	392.83	386.51	382.77	381.11	380.89	381.47	382.44	383.63	385.1	387.13	390.06	394.08	399.08	404.57	409.91	414.52	418.25	421.44	424.86	429.35	435.48	443.21	451.9	460.51	468.11	474.29	479.33	483.97	488.98	494.78	501.23	507.76	513.77	518.99	523.71	528.59	534.32	541.18	548.86	556.6	563.66	569.7	575.09	580.73	587.62	596.28	606.45	617.05	626.65	633.84	637.68
452.58	452.57	452.56	452.49	452.25	451.54	449.86	446.52	440.84	432.52	421.94	410.17	398.73	389.07	382.02	377.72	375.71	375.29	375.77	376.67	377.75	379.01	380.63	382.85	385.8	389.4	393.3	397.03	400.25	402.91	405.4	408.4	412.66	418.62	426.19	434.68	443.07	450.48	456.55	461.61	466.42	471.77	478.04	485.01	492.04	498.46	503.96	508.77	513.55	518.94	525.23	532.15	539.05	545.36	550.94	556.38	562.75	571.14	582.02	594.88	608.28	620.34	629.35	634.13
449.57	449.57	449.56	449.5	449.28	448.63	447.06	443.85	438.3	430.03	419.34	407.29	395.44	385.31	377.88	373.35	371.31	371.02	371.72	372.85	374.07	375.3	376.62	378.18	380.09	382.3	384.62	386.8	388.66	390.28	392.05	394.55	398.4	403.94	411.04	419	426.88	433.86	439.62	444.51	449.28	454.7	461.09	468.22	475.38	481.84	487.23	491.68	495.71	499.88	504.47	509.36	514.18	518.67	523.06	528.22	535.43	545.85	559.86	576.58	594.04	609.75	621.47	627.69
444.84	444.84	444.84	444.79	444.59	444	442.53	439.49	434.18	426.19	415.76	403.91	392.17	382.09	374.71	370.32	368.54	368.62	369.76	371.32	372.86	374.19	375.31	376.31	377.28	378.24	379.14	379.91	380.56	381.25	382.33	384.29	387.61	392.54	398.91	406.09	413.21	419.54	424.8	429.33	433.82	438.99	445.12	451.96	458.8	464.88	469.72	473.27	475.84	477.83	479.54	481.09	482.56	484.19	486.68	491.25	499.39	512.34	530.28	551.9	574.51	594.86	610.03	618.08
438.21	438.2	438.2	438.16	437.98	437.45	436.12	433.35	428.47	421.08	411.37	400.29	389.28	379.83	372.97	369.03	367.69	368.24	369.86	371.85	373.71	375.17	376.16	376.71	376.91	376.85	376.58	376.2	375.85	375.76	376.23	377.67	380.41	384.63	390.13	396.37	402.57	408.11	412.74	416.75	420.78	425.43	430.96	437.13	443.26	448.59	452.51	454.75	455.34	454.47	452.5	449.86	447.17	445.31	445.5	449.33	458.51	474.26	496.53	523.48	551.67	577	595.86	605.85
429.73	429.73	429.72	429.69	429.54	429.09	427.94	425.55	421.31	414.86	406.37	396.64	386.96	378.66	372.71	369.44	368.58	369.51	371.45	373.69	375.71	377.2	378.04	378.23	377.85	377.01	375.88	374.66	373.57	372.89	372.89	373.86	376.04	379.52	384.13	389.37	394.62	399.31	403.26	406.69	410.15	414.15	418.91	424.21	429.44	433.83	436.67	437.48	436	432.24	426.59	419.82	413.09	407.89	405.94	409.14	419.36	437.78	464.08	495.89	529.06	558.77	580.84	592.5
419.77	419.77	419.77	419.74	419.63	419.27	418.35	416.43	413.02	407.8	400.9	392.97	385.07	378.31	373.5	370.97	370.52	371.6	373.55	375.73	377.64	378.98	379.6	379.48	378.68	377.34	375.64	373.87	372.31	371.24	370.89	371.48	373.16	375.96	379.72	384.03	388.36	392.26	395.55	398.43	401.35	404.72	408.74	413.2	417.54	421.03	422.92	422.56	419.47	413.56	405.24	395.51	385.95	378.45	375.09	378.03	389.18	409.66	438.91	474.12	510.66	543.23	567.32	580.02
408.82	408.82	408.82	408.81	408.73	408.47	407.81	406.42	403.92	400.08	394.97	389.07	383.17	378.11	374.54	372.71	372.49	373.45	375.08	376.85	378.36	379.35	379.68	379.28	378.2	376.55	374.54	372.47	370.65	369.36	368.79	369.09	370.35	372.57	375.59	379.08	382.61	385.81	388.55	390.98	393.46	396.36	399.81	403.61	407.26	410.1	411.38	410.41	406.67	399.98	390.77	380.12	369.73	361.65	358.07	361.21	373.04	394.58	425.11	461.64	499.3	532.69	557.28	570.21
397.28	397.29	397.31	397.33	397.31	397.17	396.77	395.89	394.29	391.81	388.47	384.58	380.66	377.27	374.86	373.61	373.44	374.06	375.09	376.19	377.09	377.58	377.54	376.89	375.62	373.81	371.66	369.47	367.54	366.15	365.45	365.55	366.48	368.22	370.64	373.45	376.32	378.96	381.26	383.36	385.58	388.19	391.3	394.71	397.97	400.47	401.58	400.71	397.41	391.55	383.51	374.26	365.31	358.52	355.97	359.8	371.93	393.39	423.43	459.1	495.67	527.94	551.61	564.01
385.27	385.33	385.43	385.55	385.63	385.64	385.49	385.08	384.28	383	381.23	379.13	376.97	375.06	373.65	372.84	372.58	372.7	372.99	373.29	373.44	373.36	372.92	372.04	370.64	368.77	366.6	364.39	362.46	361.04	360.28	360.23	360.91	362.27	364.2	366.48	368.83	371.03	373.01	374.9	376.99	379.5	382.49	385.77	388.9	391.38	392.74	392.59	390.68	387	381.86	375.95	370.32	366.39	365.87	370.71	382.79	403.19	431.31	464.48	498.34	528.13	549.93	561.34
372.47	372.67	373	373.38	373.72	373.97	374.1	374.1	373.96	373.64	373.15	372.5	371.77	371.06	370.44	369.93	369.5	369.07	368.59	368.04	367.44	366.77	365.95	364.86	363.39	361.52	359.4	357.26	355.38	353.99	353.2	353.05	353.53	354.59	356.13	357.97	359.9	361.74	363.48	365.24	367.28	369.82	372.88	376.23	379.47	382.2	384.1	385.06	385.13	384.45	383.22	381.77	380.55	380.36	382.41	388.42	400.21	418.98	444.42	474.27	504.72	531.49	551.08	561.33
358.11	358.62	359.5	360.49	361.39	362.08	362.58	362.96	363.33	363.75	364.2	364.64	364.99	365.18	365.15	364.82	364.18	363.21	361.99	360.66	359.34	358.12	356.95	355.7	354.2	352.41	350.4	348.39	346.63	345.31	344.54	344.32	344.64	345.44	346.63	348.08	349.63	351.16	352.67	354.33	356.39	359.04	362.28	365.87	369.41	372.56	375.18	377.44	379.67	382.17	385.08	388.31	391.7	395.39	400.09	407.25	418.67	435.72	458.44	485.02	512.16	536.08	553.63	562.84
341.16	342.29	344.2	346.36	348.31	349.79	350.84	351.64	352.42	353.34	354.45	355.64	356.74	357.57	357.93	357.71	356.85	355.4	353.53	351.49	349.54	347.84	346.37	345.01	343.55	341.9	340.09	338.29	336.71	335.53	334.81	334.56	334.74	335.3	336.16	337.23	338.39	339.59	340.87	342.42	344.5	347.31	350.82	354.77	358.76	362.47	365.91	369.46	373.7	379.11	385.75	393.17	400.66	407.78	414.81	422.98	434.11	449.73	470.21	494.14	518.67	540.39	556.39	564.81
320.75	322.86	326.44	330.49	334.12	336.87	338.73	340.02	341.15	342.42	343.93	345.6	347.19	348.43	349.05	348.87	347.82	345.97	343.56	340.95	338.47	336.36	334.68	333.26	331.91	330.49	328.96	327.46	326.14	325.14	324.52	324.26	324.32	324.65	325.18	325.86	326.62	327.46	328.48	329.9	332.01	335.01	338.87	343.29	347.85	352.22	356.49	361.2	367.14	374.9	384.48	395.17	405.82	415.57	424.37	433.33	444.28	458.94	477.93	500.14	523.02	543.39	558.47	566.42
296.62	300.08	305.94	312.58	318.54	323.02	325.98	327.88	329.35	330.86	332.62	334.57	336.46	337.95	338.73	338.57	337.38	335.24	332.44	329.39	326.52	324.13	322.3	320.91	319.74	318.61	317.44	316.3	315.3	314.53	314.03	313.78	313.73	313.83	314.04	314.32	314.66	315.13	315.88	317.18	319.34	322.58	326.87	331.87	337.1	342.2	347.27	352.94	360.13	369.53	381.11	393.96	406.7	418.14	428.08	437.54	448.42	462.59	480.84	502.28	524.5	544.43	559.26	567.11
269.49	274.52	283.04	292.72	301.42	307.96	312.23	314.86	316.7	318.42	320.36	322.5	324.58	326.24	327.14	327.02	325.78	323.49	320.48	317.18	314.09	311.55	309.68	308.38	307.42	306.61	305.82	305.06	304.38	303.84	303.47	303.24	303.09	302.98	302.87	302.76	302.7	302.81	303.3	304.52	306.8	310.38	315.2	320.88	326.85	332.69	338.46	344.82	352.77	363.07	375.69	389.66	403.44	415.72	426.2	435.9	446.76	460.79	478.91	500.38	522.84	543.15	558.37	566.48
241.02	247.57	258.69	271.36	282.83	291.5	297.18	300.62	302.89	304.86	307	309.32	311.59	313.43	314.47	314.44	313.24	310.98	307.95	304.62	301.51	298.97	297.16	295.99	295.24	294.71	294.25	293.81	293.4	293.06	292.79	292.57	292.33	292.03	291.63	291.16	290.73	290.51	290.81	292.01	294.49	298.51	303.96	310.4	317.13	323.64	329.93	336.67	344.89	355.41	368.25	382.47	396.5	409	419.61	429.34	440.2	454.28	472.68	494.74	518.1	539.47	555.66	564.33
213.21	220.95	234.14	249.28	263.09	273.68	280.71	285.01	287.8	290.11	292.51	295.08	297.6	299.66	300.91	301.06	300.04	297.94	295.1	291.96	289.01	286.63	284.95	283.92	283.35	283.01	282.76	282.52	282.29	282.07	281.86	281.62	281.3	280.82	280.16	279.38	278.62	278.14	278.3	279.55	282.31	286.82	292.93	300.08	307.47	314.49	321.07	327.86	335.93	346.18	358.72	372.69	386.57	399.01	409.61	419.32	430.17	444.37	463.19	486.1	510.75	533.62	551.15	560.63
187.82	196.22	210.62	227.31	242.77	254.85	263.08	268.24	271.62	274.35	277.09	280	282.84	285.21	286.73	287.15	286.4	284.63	282.13	279.34	276.72	274.6	273.12	272.23	271.76	271.5	271.32	271.15	270.96	270.77	270.57	270.3	269.88	269.25	268.38	267.34	266.31	265.61	265.67	266.99	270.01	274.95	281.57	289.22	297.01	304.24	310.81	317.38	325.07	334.84	346.96	360.65	374.47	387.01	397.77	407.58	418.47	432.74	451.89	475.59	501.53	526	545.01	555.37
165.85	174.34	189.01	206.25	222.55	235.64	244.86	250.85	254.84	258.03	261.16	264.44	267.64	270.35	272.21	272.96	272.57	271.21	269.18	266.87	264.69	262.91	261.67	260.9	260.46	260.19	259.97	259.74	259.51	259.27	259.02	258.7	258.21	257.45	256.4	255.14	253.89	253.01	252.98	254.31	257.45	262.55	269.34	277.07	284.8	291.84	298.09	304.24	311.49	320.91	332.89	346.75	361.04	374.23	385.6	395.82	406.83	421.06	440.25	464.37	491.24	517	537.29	548.45
147.59	155.68	169.82	186.76	203.19	216.82	226.79	233.5	238.08	241.72	245.21	248.82	252.36	255.42	257.61	258.72	258.75	257.88	256.4	254.67	253.02	251.66	250.68	250.04	249.62	249.28	248.95	248.6	248.26	247.94	247.63	247.24	246.68	245.82	244.63	243.2	241.77	240.72	240.55	241.77	244.76	249.64	256.06	263.28	270.4	276.77	282.39	288.02	294.94	304.34	316.72	331.45	347	361.6	374.23	385.27	396.55	410.59	429.41	453.33	480.38	506.73	527.74	539.4
132.87	140.25	153.34	169.36	185.37	199.11	209.55	216.84	221.91	225.91	229.69	233.55	237.36	240.7	243.2	244.68	245.16	244.86	244.05	243.02	242	241.15	240.51	240.03	239.63	239.22	238.77	238.29	237.83	237.42	237.04	236.6	235.98	235.04	233.74	232.18	230.6	229.4	229.03	229.95	232.49	236.69	242.2	248.32	254.28	259.57	264.31	269.37	276.11	285.84	299.14	315.41	332.95	349.69	364.2	376.49	388.2	401.92	419.9	442.81	469.05	494.93	515.79	527.45
121.39	127.92	139.69	154.44	169.63	183.13	193.73	201.37	206.79	211.05	215.01	219.04	223	226.54	229.3	231.13	232.13	232.5	232.5	232.32	232.1	231.88	231.66	231.4	231.07	230.62	230.08	229.5	228.94	228.45	228.01	227.53	226.86	225.86	224.49	222.83	221.14	219.77	219.14	219.64	221.49	224.66	228.83	233.43	237.86	241.81	245.54	250.01	256.66	266.89	281.39	299.54	319.49	338.8	355.58	369.44	381.74	395.01	411.68	432.77	457.07	481.27	500.94	511.99
112.85	118.54	128.94	142.28	156.41	169.35	179.83	187.59	193.19	197.61	201.68	205.78	209.81	213.46	216.41	218.6	220.14	221.29	222.23	223.06	223.77	224.3	224.6	224.64	224.42	223.98	223.38	222.73	222.11	221.56	221.08	220.57	219.87	218.83	217.41	215.68	213.89	212.37	211.47	211.49	212.54	214.56	217.25	220.2	223.01	225.58	228.29	232.18	238.72	249.42	265.02	284.97	307.26	329.16	348.3	363.84	376.79	389.51	404.53	423.14	444.56	466.01	483.54	493.42
106.93	111.87	121.02	133.02	146.03	158.26	168.42	176.12	181.78	186.28	190.39	194.5	198.54	202.23	205.33	207.83	209.92	211.86	213.79	215.68	217.38	218.71	219.56	219.91	219.84	219.43	218.82	218.13	217.46	216.89	216.39	215.85	215.13	214.07	212.62	210.84	208.98	207.34	206.2	205.77	206.09	207.05	208.41	209.89	211.29	212.65	214.48	217.82	224.2	235.15	251.54	272.86	297.07	321.16	342.39	359.47	373.01	385.18	398.44	414.31	432.42	450.58	465.47	473.89
103.26	107.64	115.85	126.77	138.84	150.4	160.2	167.75	173.41	177.93	182.06	186.15	190.18	193.87	197.06	199.81	202.35	204.96	207.74	210.54	213.09	215.11	216.43	217.07	217.13	216.77	216.16	215.45	214.76	214.17	213.66	213.11	212.38	211.31	209.83	208.03	206.13	204.4	203.08	202.33	202.11	202.3	202.68	203.1	203.47	203.97	205.15	208.07	214.25	225.26	242.06	264.22	289.7	315.34	338.12	356.35	370.35	382.04	393.8	407.27	422.45	437.63	450.09	457.14
101.53	105.61	113.3	123.62	135.15	146.3	155.86	163.31	168.94	173.46	177.59	181.68	185.68	189.37	192.61	195.49	198.3	201.3	204.56	207.9	210.95	213.37	214.98	215.78	215.92	215.59	214.98	214.27	213.57	212.97	212.45	211.9	211.17	210.09	208.61	206.8	204.87	203.09	201.69	200.77	200.27	200.05	199.93	199.79	199.64	199.7	200.54	203.23	209.26	220.25	237.2	259.74	285.83	312.26	335.84	354.7	368.95	380.35	391.24	403.31	416.74	430.15	441.15	447.38
]'; % in deg. C
fc_extmp_map_cold=fliplr(fc_extmp_map_cold);
ex_spd=(850:50:4000)*pi/30;
fc_extmp_map_old=fc_extmp_map_cold;
fc_extmp_map_cold=zeros(size(fc_fuel_map_cold));
[a,b]=size(fc_extmp_map_cold);
for i=1:b
   fc_extmp_map_cold(:,i)=interp1(ex_spd,fc_extmp_map_old(:,i),fc_map_spd)';
end

% (g/s), engine out total exhaust flow indexed vertically by fc_map_spd and
% horizontally by fc_map_trq
fc_exflow_map_cold=[
25.9	28.2	32.26	36.97	41.14	43.98	45.66	46.99	48.8	51.41	54.55	57.86	61.08	64.21	67.12	69.68	71.86	73.95	76.21	78.68	81.25	83.77	86.26	88.61	90.65	92.21	93.43	94.55	95.83	97.3	98.8	100.15	101.21	102.02	102.64	103.08	103.32
25.85	28.11	32.09	36.73	40.87	43.73	45.48	46.87	48.7	51.3	54.41	57.7	60.91	64.07	67.03	69.65	71.89	74.02	76.3	78.77	81.32	83.82	86.31	88.67	90.76	92.44	93.83	95.11	96.47	97.89	99.25	100.44	101.38	102.12	102.69	103.11	103.34
25.69	27.81	31.58	36.01	40.05	42.99	44.93	46.51	48.46	51.07	54.14	57.38	60.6	63.82	66.88	69.61	71.96	74.16	76.48	78.95	81.47	83.94	86.41	88.78	90.95	92.81	94.46	95.98	97.45	98.8	99.97	100.93	101.7	102.31	102.81	103.17	103.37
25.38	27.23	30.54	34.55	38.41	41.49	43.8	45.79	47.99	50.69	53.74	56.95	60.19	63.49	66.67	69.56	72.05	74.37	76.75	79.21	81.69	84.13	86.57	88.94	91.17	93.19	95.08	96.84	98.42	99.72	100.72	101.48	102.09	102.58	102.98	103.26	103.42
24.92	26.35	28.99	32.36	35.91	39.19	42.04	44.64	47.26	50.14	53.21	56.42	59.7	63.1	66.43	69.49	72.16	74.61	77.04	79.5	81.93	84.33	86.73	89.1	91.37	93.49	95.54	97.45	99.11	100.39	101.31	101.96	102.47	102.87	103.17	103.37	103.47
24.39	25.36	27.22	29.85	33.02	36.47	39.9	43.17	46.29	49.41	52.54	55.78	59.12	62.64	66.15	69.41	72.25	74.83	77.32	79.77	82.15	84.49	86.87	89.22	91.5	93.67	95.79	97.77	99.47	100.74	101.65	102.29	102.76	103.11	103.33	103.46	103.52
23.93	24.48	25.64	27.56	30.33	33.84	37.71	41.55	45.12	48.49	51.73	55.02	58.45	62.13	65.83	69.28	72.27	74.96	77.48	79.92	82.27	84.59	86.94	89.28	91.54	93.72	95.83	97.8	99.48	100.78	101.73	102.42	102.93	103.26	103.44	103.52	103.55
23.62	23.88	24.55	25.94	28.32	31.73	35.78	39.95	43.85	47.44	50.81	54.19	57.74	61.59	65.48	69.11	72.22	74.96	77.49	79.93	82.28	84.61	86.96	89.28	91.5	93.62	95.66	97.56	99.21	100.53	101.57	102.37	102.95	103.31	103.49	103.55	103.57
23.45	23.56	23.95	24.99	27.04	30.23	34.23	38.51	42.58	46.34	49.82	53.29	56.97	60.99	65.06	68.84	72.03	74.78	77.31	79.77	82.17	84.55	86.92	89.24	91.41	93.43	95.35	97.13	98.71	100.07	101.21	102.16	102.86	103.29	103.49	103.55	103.57
23.35	23.41	23.68	24.52	26.32	29.27	33.1	37.31	41.4	45.18	48.69	52.19	55.95	60.12	64.37	68.29	71.56	74.35	76.92	79.44	81.92	84.39	86.83	89.16	91.3	93.25	95.03	96.67	98.17	99.53	100.77	101.85	102.68	103.19	103.43	103.5	103.52
23.24	23.28	23.51	24.25	25.89	28.64	32.27	36.31	40.25	43.89	47.26	50.67	54.42	58.67	63.06	67.1	70.48	73.38	76.09	78.78	81.45	84.07	86.62	89.04	91.22	93.14	94.85	96.38	97.78	99.1	100.37	101.52	102.43	103	103.27	103.36	103.37
23.09	23.12	23.33	24.02	25.54	28.11	31.52	35.31	38.97	42.31	45.38	48.56	52.19	56.43	60.87	65	68.5	71.59	74.55	77.55	80.52	83.4	86.17	88.75	91.07	93.08	94.81	96.31	97.64	98.89	100.1	101.23	102.13	102.7	102.98	103.07	103.09
22.88	22.91	23.1	23.72	25.11	27.46	30.58	34.04	37.36	40.35	43.09	46	49.43	53.53	57.89	62.01	65.6	68.9	72.2	75.6	78.96	82.19	85.27	88.14	90.7	92.88	94.73	96.29	97.63	98.82	99.94	100.96	101.78	102.3	102.55	102.64	102.65
22.63	22.66	22.82	23.35	24.55	26.59	29.34	32.42	35.41	38.13	40.65	43.36	46.59	50.48	54.65	58.64	62.25	65.73	69.33	73.08	76.81	80.42	83.88	87.1	89.95	92.35	94.35	96.02	97.43	98.63	99.71	100.66	101.4	101.86	102.08	102.15	102.16
22.38	22.4	22.53	22.95	23.93	25.64	27.98	30.68	33.4	35.97	38.42	41.05	44.13	47.8	51.72	55.54	59.09	62.64	66.41	70.39	74.41	78.36	82.19	85.75	88.86	91.43	93.53	95.28	96.78	98.09	99.25	100.24	100.99	101.44	101.65	101.71	101.72
22.15	22.16	22.26	22.6	23.39	24.8	26.78	29.15	31.66	34.16	36.63	39.26	42.26	45.74	49.44	53.06	56.5	60.01	63.81	67.89	72.09	76.34	80.52	84.4	87.69	90.31	92.4	94.15	95.72	97.17	98.52	99.69	100.57	101.09	101.33	101.4	101.42
21.93	21.94	22.02	22.32	23	24.21	25.93	28.05	30.39	32.83	35.3	37.92	40.85	44.19	47.69	51.11	54.37	57.76	61.49	65.62	70.03	74.61	79.16	83.33	86.75	89.36	91.35	93.02	94.58	96.14	97.69	99.08	100.15	100.8	101.1	101.2	101.21
21.72	21.73	21.81	22.09	22.72	23.83	25.39	27.32	29.5	31.84	34.25	36.83	39.69	42.89	46.22	49.44	52.49	55.68	59.31	63.5	68.16	73.13	78.09	82.57	86.16	88.78	90.71	92.29	93.81	95.39	97.04	98.58	99.8	100.56	100.91	101.02	101.05
21.49	21.5	21.59	21.86	22.47	23.51	24.96	26.75	28.78	30.99	33.31	35.8	38.57	41.66	44.85	47.88	50.72	53.72	57.25	61.49	66.4	71.73	77.1	81.93	85.77	88.53	90.53	92.12	93.61	95.15	96.77	98.31	99.55	100.34	100.71	100.83	100.86
21.2	21.21	21.29	21.55	22.13	23.1	24.45	26.12	28.03	30.12	32.34	34.76	37.45	40.46	43.54	46.44	49.12	51.96	55.39	59.64	64.66	70.23	75.92	81.1	85.28	88.34	90.55	92.27	93.79	95.29	96.81	98.25	99.41	100.16	100.52	100.63	100.65
20.81	20.82	20.89	21.13	21.65	22.54	23.8	25.37	27.18	29.19	31.33	33.67	36.3	39.26	42.3	45.13	47.72	50.46	53.79	57.95	62.93	68.53	74.38	79.86	84.44	87.89	90.42	92.34	93.95	95.44	96.89	98.23	99.32	100.01	100.35	100.46	100.48
20.28	20.29	20.34	20.54	21.01	21.82	22.99	24.49	26.23	28.17	30.24	32.52	35.11	38.05	41.07	43.88	46.44	49.12	52.35	56.35	61.15	66.61	72.5	78.22	83.19	87.05	89.9	92.01	93.71	95.23	96.7	98.07	99.17	99.88	100.22	100.34	100.36
19.58	19.58	19.63	19.81	20.22	20.97	22.09	23.53	25.22	27.1	29.09	31.3	33.83	36.75	39.77	42.57	45.12	47.76	50.92	54.76	59.36	64.65	70.51	76.39	81.66	85.82	88.88	91.09	92.86	94.46	96.06	97.6	98.87	99.7	100.11	100.26	100.29
18.66	18.67	18.72	18.88	19.28	20.02	21.11	22.53	24.18	25.99	27.91	30.04	32.52	35.39	38.37	41.14	43.66	46.27	49.39	53.17	57.69	62.92	68.77	74.73	80.1	84.33	87.41	89.62	91.39	93.09	94.9	96.72	98.28	99.35	99.92	100.16	100.23
17.57	17.57	17.62	17.79	18.21	18.96	20.08	21.49	23.11	24.88	26.76	28.84	31.25	34.05	36.95	39.66	42.13	44.72	47.84	51.67	56.29	61.64	67.57	73.5	78.73	82.78	85.66	87.73	89.46	91.23	93.23	95.34	97.23	98.63	99.49	99.93	100.12
16.39	16.38	16.41	16.56	16.99	17.77	18.92	20.36	21.99	23.76	25.63	27.68	30.04	32.75	35.56	38.19	40.6	43.16	46.32	50.31	55.19	60.83	66.91	72.76	77.69	81.35	83.9	85.75	87.38	89.13	91.18	93.43	95.59	97.38	98.68	99.5	99.89
15.34	15.29	15.25	15.35	15.72	16.48	17.63	19.09	20.76	22.55	24.42	26.47	28.8	31.45	34.19	36.76	39.12	41.67	44.87	49.01	54.16	60.11	66.37	72.17	76.82	80.11	82.35	84	85.48	87.09	88.98	91.15	93.46	95.65	97.51	98.85	99.55
14.59	14.48	14.32	14.27	14.5	15.16	16.25	17.71	19.38	21.19	23.07	25.11	27.43	30.07	32.79	35.33	37.68	40.21	43.41	47.61	52.89	59.03	65.43	71.26	75.82	78.97	81.08	82.65	84.03	85.41	86.99	88.9	91.2	93.74	96.16	98.07	99.11
14.21	14.01	13.69	13.43	13.46	13.95	14.95	16.35	18.02	19.82	21.69	23.72	26.04	28.69	31.43	34	36.35	38.85	42	46.12	51.31	57.39	63.83	69.76	74.48	77.77	80	81.63	82.98	84.17	85.42	87.03	89.23	91.98	94.82	97.16	98.47
14.09	13.8	13.32	12.85	12.67	13	13.88	15.24	16.88	18.66	20.51	22.53	24.85	27.52	30.3	32.89	35.25	37.74	40.8	44.72	49.65	55.48	61.84	67.91	72.91	76.49	78.96	80.72	82.11	83.24	84.31	85.71	87.76	90.49	93.45	95.94	97.34
14.09	13.72	13.1	12.48	12.15	12.37	13.2	14.52	16.14	17.9	19.73	21.74	24.06	26.75	29.56	32.17	34.54	37	39.97	43.68	48.29	53.81	60.03	66.18	71.44	75.27	77.89	79.74	81.18	82.35	83.44	84.76	86.64	89.14	91.84	94.13	95.43
14.1	13.69	12.98	12.26	11.87	12.06	12.88	14.19	15.79	17.53	19.34	21.34	23.66	26.37	29.18	31.81	34.19	36.63	39.53	43.08	47.45	52.72	58.82	64.99	70.37	74.28	76.88	78.66	80.1	81.4	82.64	84	85.69	87.75	89.91	91.72	92.75
14.1	13.66	12.91	12.15	11.75	11.97	12.8	14.12	15.7	17.42	19.22	21.21	23.53	26.24	29.06	31.69	34.07	36.5	39.37	42.84	47.08	52.21	58.22	64.38	69.76	73.6	76.05	77.67	79.07	80.48	81.91	83.35	84.85	86.44	87.99	89.26	89.97
14.09	13.64	12.87	12.11	11.73	11.96	12.82	14.14	15.71	17.41	19.19	21.18	23.51	26.21	29.03	31.66	34.04	36.47	39.33	42.77	46.97	52.05	58.03	64.17	69.51	73.27	75.58	77.07	78.42	79.9	81.46	82.96	84.33	85.61	86.76	87.67	88.17
]';
fc_exflow_map_cold=fliplr(fc_exflow_map_cold);
% End of cold maps
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%5

%Process Cold Maps to generate Correction Factor Maps
names={'fc_fuel_map','fc_hc_map','fc_co_map','fc_nox_map','fc_pm_map'};
for i=1:length(names)
    %cold to hot raio, e.g. fc_fuel_map_c2h = fc_fuel_map_cold ./ fc_fuel_map
    eval([names{i},'_c2h=',names{i},'_cold./',names{i},';'])
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% LIMITS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (N*m), max torque curve of the engine indexed by fc_map_spd (from Sluder printout 12/14/98) (Note 4400 was used rather than 4381, because that is the value in the table)
temp_max_speed=[0 899 1194 1593 1892 2389 2788 3186 3584 3982 4400]*pi/30; %note, max torque at idle (900 RPM) was taken as a % of torque at 1200 RPM (last real value)
temp_max_torque=[0 0.8*112.3 112.3 148.1 160.3 152.3 146.4 138.2 130.2 119.1 76.1]*(1/(3.281*0.224809));
fc_max_trq=interp1(temp_max_speed,temp_max_torque,fc_map_spd,'linear');%	N*m
clear temp_max_speed temp_max_torque

% (N*m), closed throttle torque of the engine (max torque that can be absorbed)
% indexed by fc_map_spd -- correlation from JDMA
fc_ct_trq=4.448/3.281*(-fc_disp)*61.02/24 * ...
   (9*(fc_map_spd/max(fc_map_spd)).^2 + 14 * (fc_map_spd/max(fc_map_spd)));


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DEFAULT SCALING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% (--), used to scale fc_map_spd to simulate a faster or slower running engine 
fc_spd_scale=1.0;
% (--), used to scale fc_map_trq to simulate a higher or lower torque engine
fc_trq_scale=1.0;
fc_pwr_scale=fc_spd_scale*fc_trq_scale;   % --  scale fc power


% user definable mass scaling function
fc_mass_scale_fun=inline('(x(1)*fc_trq_scale+x(2))*(x(3)*fc_spd_scale+x(4))*(fc_base_mass+fc_acc_mass)+fc_fuel_mass','x','fc_spd_scale','fc_trq_scale','fc_base_mass','fc_acc_mass','fc_fuel_mass');
fc_mass_scale_coef=[1 0 1 0]; % coefficients of mass scaling function


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% STUFF THAT SCALES WITH TRQ & SPD SCALES (MASS AND INERTIA)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fc_inertia=(1.9/(0.224809*12*3.28))*fc_pwr_scale;   % (kg*m^2), =0.2147, rotational inertia of the engine 
                                                            %            (estimate from ORNL)
fc_max_pwr=(max(fc_map_spd.*fc_max_trq)/1000)*fc_pwr_scale; % kW     peak engine power

fc_base_mass=1.8*fc_max_pwr;            % (kg), mass of the engine block and head (base engine)
                                        %       assuming a mass penalty of 1.8 kg/kW from S. Sluder (ORNL) estimate of 300 lb 
fc_acc_mass=0.8*fc_max_pwr;             % kg    engine accy's, electrics, cntrl's - assumes mass penalty of 0.8 kg/kW (from 1994 OTA report, Table 3)
fc_fuel_mass=0.6*fc_max_pwr;            % kg    mass of fuel and fuel tank (from 1994 OTA report, Table 3)
fc_mass=fc_base_mass+fc_acc_mass+fc_fuel_mass; % kg  total engine/fuel system mass
fc_ext_sarea=0.3*(fc_max_pwr/100)^0.67;       % m^2    exterior surface area of engine


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% OTHER DATA
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fc_fuel_den=843;                        % (g/l), density of the fuel from Scott Sluder (ORNL, 1/19/99)
% fc_fuel_den=835;                      % (g/l), density of the fuel (from original TDI map from publication) 
% fc_fuel_den=850;                      % (g/l), density of fuel from handbook
fc_fuel_lhv=18456*2.205/0.9478;         % (J/g), =42937, lower heating value of the fuel from Scott Sluder (ORNL, 1/19/99)
% fc_fuel_lhv=43000;                    % (J/g), lower heating value of the fuel from handbook

%the following was added for the new thermal modeling of the engine 12/17/98 ss and sb
fc_tstat=99;                  % C      engine coolant thermostat set temperature (typically 95 +/- 5 C)
fc_cp=500;                    % J/kgK  ave cp of engine (iron=500, Al or Mg = 1000)
fc_h_cp=500;                  % J/kgK  ave cp of hood & engine compartment (iron=500, Al or Mg = 1000)
fc_hood_sarea=1.5;            % m^2    surface area of hood/eng compt.
fc_emisv=0.8;                 %        eff emissivity of engine ext surface to hood int surface
fc_hood_emisv=0.9;            %        emissivity hood ext
fc_h_air_flow=0.0;            % kg/s   heater air flow rate (140 cfm=0.07)
fc_cl2h_eff=0.7;              % --     ave cabin heater HX eff (based on air side)
fc_c2i_th_cond=500;           % W/K    conductance btwn engine cyl & int
fc_i2x_th_cond=500;           % W/K    conductance btwn engine int & ext
fc_h2x_th_cond=10;            % W/K    conductance btwn engine & engine compartment


%the following variable is not used directly in modelling and should always be equal to one
%it's used for initialization purposes
fc_eff_scale=1;

% clean up workspace
clear fc_exflow_map_gpkWh fc_co2_map_gpkWh fc_map_kW T w

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% REVISION HISTORY
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 05/19/98 (tm): file created from data in a_vw19l.m
% 07/03/98 (MC): changed fc_init_coolant_temp to fc_coolant_init_temp
% 07/06/98 (MC): corrected max power computation in mass calc.
% 07/07/98 (MC): cosmetic change
% 07/16/98 (SS): added variable fc_fuel_type under file id section
% 07/17/98 (tm): renamed FC_CI66.M for consistency
% 08/28/98 (MC): added variable fc_disp under file id section
%                fc_ct_trq computed according to correlation from JDMA, 5/98
% 10/07/98 (MC): reduced extrapolated torques in fc_max_trq
% 10/9/98 (vh,sb,ss): added pm and removed init conditions and added new exhaust variables
% 10/13/98 (MC): updated equation for fc_ct_trq (convert from ft-lb to Nm)
% 12/17/98 ss,sb: added 12 new variables for engine thermal modelling.
% 01/25/99 (SB): modified thermal section to work with new BD, revised FC mass calc's

% 01/05/99 (KW): used FC_CI66 as base file in which to replace data
%					  with ORNL data
% 01/14/99 (SB): removed unneeded variables (fc_air_fuel_ratio, fc_ex_pwr_frac)
% 2/4/99: (ss,sb) changed fc_ext_sarea=0.3*(fc_max_pwr/100)^0.67  it was 0.3*(fc_max_pwr/100)
%		it now takes into account that surface area increases based on mass to the 2/3 power 
% 3/15/99:ss updated *_version to 2.1 from 2.0
% 6/4/99: (KW): source data obtained in g/s data, rather than g/kWh
% 6/7/99: (KW): updated fuel map (g/s) obtained from ORNL
% 7/7/99: (KW): updated comments and removed previous g/kWh conversion comments
% 7/7/99:tm added conversions to g/kWh for plotting purposes
% 11/03/99:ss updated version from 2.2 to 2.21
% 12/30/99:ss interp1 was INTERP1 (changed to work in unix)
% 01/17/010: vhj added variables fc_o2_map, fc_exflow_vol_map (to be used in emissions modeling)
% 01/31/01: vhj added cold maps and correction factor map calculations (new variables e.g. fc_fuel_map_c2h)
%               added boolean fc_cold, 0=no cold data; 1=cold data exists
%02/26/01: vhj updated O2 map (was missing a column)
% 7/30/01:tm added user definable mass scaling function mass=f(fc_spd_scale,fc_trq_scale,fc_base_mass,fc_acc_mass,fc_fuel_mass)
